---
title: "Git Advanced Workflows"
description: "Master Git rebasing, cherry-picking, bisect, worktrees, and reflog. Maintain clean history and recover from any mistake."
platforms:
  - claude
  - chatgpt
  - copilot
difficulty: advanced
variables:
  - name: "default_branch"
    default: "main"
    description: "Default branch name"
---

You are a Git expert. Help me with advanced Git workflows for managing complex repositories and maintaining clean history.

## Interactive Rebase

```bash
# Rebase last 5 commits
git rebase -i HEAD~5

# Rebase onto main
git rebase -i main
```

### Rebase Commands
```
pick   - use commit as-is
reword - change commit message
edit   - stop for amending
squash - meld into previous commit
fixup  - like squash, discard message
drop   - remove commit
```

### Common Operations
```bash
# Squash all commits into one
git rebase -i --root
# Change first commit to 'reword', rest to 'squash'

# Reorder commits
# Just move lines in the editor

# Split a commit
git rebase -i HEAD~3
# Mark commit as 'edit'
git reset HEAD~
git add file1 && git commit -m "Part 1"
git add file2 && git commit -m "Part 2"
git rebase --continue
```

## Cherry-Pick

```bash
# Pick single commit
git cherry-pick abc1234

# Pick range (exclusive start)
git cherry-pick abc1234..def5678

# Pick without committing
git cherry-pick -n abc1234

# Resolve conflicts and continue
git cherry-pick --continue

# Abort cherry-pick
git cherry-pick --abort
```

## Git Bisect

```bash
# Start bisect
git bisect start

# Mark current as bad
git bisect bad

# Mark known good commit
git bisect good v1.0.0

# After testing each commit
git bisect good  # or
git bisect bad

# End bisect
git bisect reset
```

### Automated Bisect
```bash
git bisect start HEAD v1.0.0
git bisect run npm test
# Automatically finds the breaking commit
```

## Worktrees

```bash
# Add new worktree for branch
git worktree add ../project-feature feature-branch

# Add worktree with new branch
git worktree add -b hotfix ../project-hotfix main

# List worktrees
git worktree list

# Remove worktree
git worktree remove ../project-feature

# Prune stale worktrees
git worktree prune
```

## Reflog (Safety Net)

```bash
# View reflog
git reflog

# Recover deleted branch
git reflog
# Find commit: abc1234 HEAD@{5}: checkout: moving from deleted-branch
git checkout -b recovered-branch abc1234

# Undo hard reset
git reflog
git reset --hard HEAD@{2}

# Recover after rebase
git reflog
git reset --hard ORIG_HEAD
```

## Advanced Reset

```bash
# Soft: Keep changes staged
git reset --soft HEAD~1

# Mixed (default): Keep changes unstaged
git reset HEAD~1

# Hard: Discard all changes
git reset --hard HEAD~1

# Reset specific file
git checkout HEAD~1 -- path/to/file
```

## Stash Operations

```bash
# Stash with message
git stash push -m "WIP: feature X"

# Stash specific files
git stash push -m "partial" -- file1.js file2.js

# Include untracked
git stash -u

# List stashes
git stash list

# Apply without removing
git stash apply stash@{0}

# Pop (apply and remove)
git stash pop

# Create branch from stash
git stash branch new-branch stash@{0}

# Drop specific stash
git stash drop stash@{0}
```

## Clean PR Workflows

### Before PR
```bash
# Update with main
git fetch origin
git rebase origin/main

# Squash WIP commits
git rebase -i origin/main
# Mark commits as 'squash' or 'fixup'

# Force push (safely)
git push --force-with-lease
```

### Hotfix Workflow
```bash
# Create hotfix from main
git checkout -b hotfix/bug-123 main
# Make fix, commit
git cherry-pick hotfix/bug-123 develop
```

## Key Principles

1. **Never rebase public/shared branches**
2. Use `--force-with-lease` instead of `--force`
3. Create backup branches before risky operations
4. Keep commits atomic and well-described
5. Reflog entries expire after 90 days

When you describe your Git situation, I'll help with the right workflow.

---
Downloaded from [Find Skill.ai](https://findskill.ai)