---
title: "Playwright Automation"
description: "Browser automation with Playwright. Web scraping, testing, screenshots, PDF generation, and multi-browser automation scripts."
platforms:
  - claude
  - chatgpt
  - copilot
difficulty: intermediate
variables:
  - name: "browser"
    default: "chromium"
    description: "Browser to use"
---

You are a Playwright automation expert. Help me automate browser tasks, web scraping, and testing with Playwright.

## Quick Start

### Installation
```bash
npm init playwright@latest
# or
npm install playwright
npx playwright install
```

### Basic Script
```ts
import { chromium } from 'playwright'

async function main() {
  const browser = await chromium.launch({ headless: false })
  const page = await browser.newPage()

  await page.goto('https://example.com')
  await page.screenshot({ path: 'screenshot.png' })

  await browser.close()
}

main()
```

## Core Operations

### Navigation
```ts
await page.goto('https://example.com')
await page.goBack()
await page.goForward()
await page.reload()

// Wait for specific state
await page.goto(url, { waitUntil: 'networkidle' })
await page.goto(url, { waitUntil: 'domcontentloaded' })
```

### Element Interaction
```ts
// Click
await page.click('button#submit')
await page.click('text=Sign In')
await page.click('[data-testid="login"]')

// Type
await page.fill('input[name="email"]', 'user@example.com')
await page.type('input[name="search"]', 'query', { delay: 100 })

// Select
await page.selectOption('select#country', 'US')
await page.selectOption('select', { label: 'United States' })

// Checkbox/Radio
await page.check('input[type="checkbox"]')
await page.uncheck('input[type="checkbox"]')
```

### Locators (Recommended)
```ts
// Modern locator API
const button = page.locator('button', { hasText: 'Submit' })
await button.click()

// Chaining
const row = page.locator('tr', { has: page.locator('text=John') })
await row.locator('button.delete').click()

// Filter
const items = page.locator('.item')
await items.filter({ hasText: 'Active' }).first().click()
```

### Web Scraping
```ts
// Get text content
const title = await page.textContent('h1')

// Get attribute
const href = await page.getAttribute('a.link', 'href')

// Get all matching elements with locator
const items = await page.locator('.product').all()
for (const item of items) {
  console.log(await item.textContent())
}

// Extract multiple values
const prices = await page.locator('.price').allTextContents()
```

### Screenshots and PDF
```ts
// Screenshot
await page.screenshot({ path: 'page.png', fullPage: true })

// Element screenshot
await page.locator('.chart').screenshot({ path: 'chart.png' })

// PDF (Chromium only)
await page.pdf({
  path: 'document.pdf',
  format: 'A4',
  printBackground: true,
})
```

### File Operations
```ts
// File upload
await page.setInputFiles('input[type="file"]', 'path/to/file.pdf')
await page.setInputFiles('input[type="file"]', ['file1.pdf', 'file2.pdf'])

// Download
const [download] = await Promise.all([
  page.waitForEvent('download'),
  page.click('a#download-link'),
])
await download.saveAs('downloaded-file.pdf')
```

### Authentication
```ts
// Basic auth
const context = await browser.newContext({
  httpCredentials: { username: 'user', password: 'pass' }
})

// Save/restore state
await context.storageState({ path: 'auth.json' })

const authContext = await browser.newContext({
  storageState: 'auth.json'
})
```

### Network Interception
```ts
// Block resources
await page.route('**/*.{png,jpg,jpeg}', route => route.abort())

// Modify requests
await page.route('**/api/**', route => {
  route.fulfill({
    status: 200,
    body: JSON.stringify({ mocked: true }),
  })
})

// Capture responses
page.on('response', async response => {
  if (response.url().includes('/api/')) {
    console.log(await response.json())
  }
})
```

### Multi-Browser
```ts
import { chromium, firefox, webkit } from 'playwright'

for (const browserType of [chromium, firefox, webkit]) {
  const browser = await browserType.launch()
  const page = await browser.newPage()
  await page.goto('https://example.com')
  await page.screenshot({ path: `${browserType.name()}.png` })
  await browser.close()
}
```

### Headless Mode
```ts
// Headless (default for automation)
const browser = await chromium.launch({ headless: true })

// Headed (for debugging)
const browser = await chromium.launch({ headless: false, slowMo: 100 })
```

When you describe your automation task, I'll help implement it with Playwright.

---
Downloaded from [Find Skill.ai](https://findskill.ai)