---
name: test-suite-generator
version: 1.0.0
---

# Initialization

Generate comprehensive test suites for any codebase. Analyzes existing code and creates unit tests, integration tests, and e2e tests with proper coverage across Jest, Pytest, JUnit, and more.

## Structure

This skill contains:
- `SKILL.md` - Main skill prompt with testing patterns and framework examples

## Files to Generate

None (prompt-only skill - copy SKILL.md content directly)

## Dependencies

None required. The skill adapts to your testing framework:
- **Python**: pytest, pytest-mock, pytest-cov
- **JavaScript/TypeScript**: jest, vitest, @testing-library/react
- **Java**: junit-jupiter, mockito
- **Go**: testing package, testify

## Post-Init Steps

### Claude Code
```bash
# Copy to your skills directory
cp -r test-suite-generator/ ~/.claude/skills/test-suite-generator/
```

### Other AI Assistants
1. Open `SKILL.md`
2. Copy the content after the frontmatter (after the second `---`)
3. Paste into your AI assistant's system prompt or chat

## Usage Examples

### Generate Tests for Python Code
```
Analyze this Python service and generate pytest tests:

class OrderService:
    def __init__(self, db, payment_gateway):
        self.db = db
        self.payment_gateway = payment_gateway

    def create_order(self, user_id, items):
        # Validate items
        if not items:
            raise ValueError("Order must have at least one item")

        total = sum(item['price'] * item['quantity'] for item in items)

        # Create order in database
        order = self.db.orders.create(user_id=user_id, items=items, total=total)

        return order

    def process_payment(self, order_id, payment_method):
        order = self.db.orders.find(order_id)
        if not order:
            raise OrderNotFoundError(f"Order {order_id} not found")

        result = self.payment_gateway.charge(
            amount=order.total,
            method=payment_method
        )

        if result.success:
            self.db.orders.update(order_id, status='paid')

        return result
```

### Generate Tests for React Component
```
Generate Vitest tests with Testing Library for this component:

function UserProfile({ userId }) {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    fetchUser(userId)
      .then(setUser)
      .catch(setError)
      .finally(() => setLoading(false));
  }, [userId]);

  if (loading) return <Spinner />;
  if (error) return <ErrorMessage message={error.message} />;

  return (
    <div className="user-profile">
      <Avatar src={user.avatar} alt={user.name} />
      <h1>{user.name}</h1>
      <p>{user.email}</p>
    </div>
  );
}
```

### Generate Tests for Go Service
```
Generate Go tests with testify for this handler:

type UserHandler struct {
    repo UserRepository
}

func (h *UserHandler) GetUser(w http.ResponseWriter, r *http.Request) {
    id := chi.URLParam(r, "id")
    userID, err := strconv.ParseInt(id, 10, 64)
    if err != nil {
        http.Error(w, "Invalid user ID", http.StatusBadRequest)
        return
    }

    user, err := h.repo.FindByID(r.Context(), userID)
    if err != nil {
        if errors.Is(err, ErrNotFound) {
            http.Error(w, "User not found", http.StatusNotFound)
            return
        }
        http.Error(w, "Internal error", http.StatusInternalServerError)
        return
    }

    json.NewEncoder(w).Encode(user)
}
```

## Compatibility

Tested with: claude, chatgpt, gemini, copilot

## Variables

Customize these placeholders in the skill:

| Variable | Default | Description |
|----------|---------|-------------|
| `{{testing_framework}}` | `auto-detect` | Testing framework (Jest, Pytest, JUnit, etc.) |
| `{{coverage_target}}` | `80` | Target code coverage percentage |
| `{{test_types}}` | `unit,integration` | Types of tests to generate |
| `{{mocking_strategy}}` | `minimal` | How aggressively to mock dependencies |
| `{{code_language}}` | `auto-detect` | Primary programming language |

## What You'll Get

- Complete test files ready to run
- Proper test organization and naming
- Mocking strategies for external dependencies
- Edge case and error handling tests
- Coverage analysis and recommendations
- Framework-specific best practices

## Tips for Best Results

1. **Provide context**: Include imports and dependencies
2. **Specify framework**: If not auto-detectable, state your testing framework
3. **Mention external services**: Databases, APIs, etc. that need mocking
4. **Set coverage goals**: State if you need specific coverage targets
5. **Include types**: TypeScript types or Python type hints help generate better tests

---
Downloaded from [Find Skill.ai](https://findskill.ai)
