---
title: "Testing Anti-Patterns"
description: "Identify and prevent common testing mistakes. Stop testing mock behavior, avoid production pollution, and write meaningful tests."
platforms:
  - claude
  - chatgpt
  - copilot
difficulty: intermediate
variables:
  - name: "framework"
    default: "jest"
    description: "Test framework"
---

You are a testing quality expert. Help me identify and fix common testing anti-patterns.

## The Iron Laws of Testing

1. **Never test mock behavior**
2. **Never add test-only methods to production code**
3. **Never mock without understanding dependencies**

> Test what the code does, not what the mocks do.

## Anti-Pattern 1: Testing Mock Behavior

### Bad Example
```ts
// Testing the mock, not the code!
it('should call the API', () => {
  const mockApi = jest.fn().mockReturnValue({ data: 'test' })
  const service = new Service(mockApi)

  service.fetchData()

  expect(mockApi).toHaveBeenCalled()  // Testing mock behavior
  expect(mockApi).toHaveBeenCalledWith('/endpoint')  // Still testing mock
})
```

### Good Example
```ts
// Testing actual behavior
it('should return formatted data', () => {
  const mockApi = jest.fn().mockReturnValue({ data: 'test' })
  const service = new Service(mockApi)

  const result = service.fetchData()

  expect(result).toEqual({ formatted: 'TEST' })  // Testing real output
})
```

## Anti-Pattern 2: Test-Only Methods in Production

### Bad Example
```ts
// Production code polluted with test helpers
class UserService {
  private cache = new Map()

  getUser(id: string) { /* ... */ }

  // DON'T DO THIS
  _clearCacheForTesting() {
    this.cache.clear()
  }
}
```

### Good Example
```ts
// Clean production code
class UserService {
  constructor(private cache: Cache) {}
  getUser(id: string) { /* ... */ }
}

// Test setup handles isolation
beforeEach(() => {
  const testCache = new InMemoryCache()
  service = new UserService(testCache)
})
```

## Anti-Pattern 3: Mocking Without Understanding

### Bad Example
```ts
// Mocking without understanding side effects
jest.mock('./emailService')

it('should register user', async () => {
  await userService.register({ email: 'test@test.com' })
  expect(User.create).toHaveBeenCalled()
  // Forgot email service validation, confirmation, etc.
})
```

### Good Example
```ts
// Understand the full dependency chain
it('should register user and send confirmation', async () => {
  const mockEmail = jest.spyOn(emailService, 'sendConfirmation')

  await userService.register({ email: 'test@test.com' })

  expect(User.create).toHaveBeenCalled()
  expect(mockEmail).toHaveBeenCalledWith('test@test.com', expect.any(String))
})
```

## Anti-Pattern 4: Incomplete Mocks

### Bad Example
```ts
// Partial mock hides assumptions
const mockResponse = { data: { id: 1 } }  // Missing fields!

// Later code breaks because it expects:
// { data: { id: 1, name: 'Test', status: 'active', createdAt: '...' } }
```

### Good Example
```ts
// Complete mock mirrors real API
const mockResponse = {
  data: {
    id: 1,
    name: 'Test User',
    status: 'active',
    createdAt: '2024-01-01T00:00:00Z',
  },
  meta: { requestId: 'test-123' },
}
```

## Anti-Pattern 5: Integration Tests as Afterthought

### Bad Approach
```
1. Write code
2. Write unit tests
3. "I'll add integration tests later"
4. Never adds integration tests
5. Bug in production
```

### Good Approach (TDD)
```
1. Write failing test
2. Write minimal code to pass
3. Refactor
4. Repeat
```

## Self-Check Questions

Before committing tests, ask:

1. **Am I testing the mock or the code?**
   - Remove the mock. Does the test still make sense?

2. **Would this test catch a real bug?**
   - If the implementation changes but behavior is correct, should the test fail?

3. **Am I adding code only for tests?**
   - Test through public APIs, not test backdoors

4. **Do I understand what I'm mocking?**
   - Can I explain the real behavior?

5. **Are my mocks complete?**
   - Do they match the real API response shape?

## Better Alternatives

| Instead of... | Try... |
|

---
Downloaded from [Find Skill.ai](https://findskill.ai)