---
title: "Healthcare Analytics"
description: "Analyze healthcare data for clinical outcomes, operational efficiency, and population health insights."
platforms:
  - claude
  - chatgpt
  - gemini
difficulty: advanced
variables:
  - name: "analytics_focus"
    default: "outcomes"
    description: "Healthcare analytics focus"
---

You are a healthcare analytics expert. Help me analyze healthcare data to improve outcomes and operations.

## Healthcare Analytics Framework

### Key Metric Categories
```
CLINICAL OUTCOMES
├── Mortality rates
├── Readmission rates
├── Complication rates
├── Length of stay
└── Patient safety indicators

OPERATIONAL METRICS
├── Bed occupancy
├── Wait times
├── Throughput
├── Staff utilization
└── Appointment no-show rate

FINANCIAL METRICS
├── Cost per case
├── Revenue per patient
├── Collection rate
├── Denial rate
└── Days in A/R

QUALITY METRICS
├── HEDIS measures
├── Patient satisfaction
├── Core measures compliance
├── Infection rates
└── Fall rates
```

## Clinical Outcomes Analysis

### Readmission Analysis
```python
def analyze_readmissions(df, days=30):
    """
    Analyze hospital readmission rates

    df: Patient encounter data
    """

    # Calculate readmission flag
    df = df.sort_values(['patient_id', 'admission_date'])

    df['prev_discharge'] = df.groupby('patient_id')['discharge_date'].shift(1)
    df['days_since_discharge'] = (
        df['admission_date'] - df['prev_discharge']
    ).dt.days

    df['is_readmission'] = df['days_since_discharge'] <= days

    # Overall readmission rate
    readmission_rate = df['is_readmission'].mean() * 100

    # By diagnosis
    by_diagnosis = df.groupby('primary_diagnosis').agg({
        'is_readmission': ['sum', 'mean', 'count']
    })
    by_diagnosis.columns = ['readmissions', 'rate', 'total_admissions']
    by_diagnosis['rate'] = by_diagnosis['rate'] * 100

    # Risk factors
    risk_factors = df.groupby('is_readmission').agg({
        'age': 'mean',
        'length_of_stay': 'mean',
        'comorbidity_count': 'mean'
    })

    return {
        'overall_rate': readmission_rate,
        'by_diagnosis': by_diagnosis.sort_values('rate', ascending=False),
        'risk_factors': risk_factors
    }
```

### Length of Stay Analysis
```python
def analyze_los(df, benchmark=None):
    """
    Analyze length of stay patterns
    """

    # Basic statistics
    stats = {
        'mean_los': df['length_of_stay'].mean(),
        'median_los': df['length_of_stay'].median(),
        'std_los': df['length_of_stay'].std(),
        'percentile_75': df['length_of_stay'].quantile(0.75),
        'percentile_90': df['length_of_stay'].quantile(0.90)
    }

    # By diagnosis
    by_drg = df.groupby('drg').agg({
        'length_of_stay': ['mean', 'median', 'count']
    })

    # Compare to benchmark
    if benchmark is not None:
        los_index = stats['mean_los'] / benchmark * 100
        stats['los_index'] = los_index

    # Identify outliers
    outlier_threshold = df['length_of_stay'].quantile(0.95)
    outliers = df[df['length_of_stay'] > outlier_threshold]

    return {
        'statistics': stats,
        'by_drg': by_drg,
        'outlier_cases': len(outliers),
        'outlier_threshold': outlier_threshold
    }
```

## Operational Analytics

### Capacity Analysis
```python
def analyze_capacity(df, total_beds):
    """
    Analyze bed utilization and capacity
    """

    # Daily census
    daily_census = df.groupby('census_date')['patient_count'].sum()

    # Occupancy rate
    daily_occupancy = daily_census / total_beds * 100

    # Capacity metrics
    metrics = {
        'avg_occupancy': daily_occupancy.mean(),
        'peak_occupancy': daily_occupancy.max(),
        'min_occupancy': daily_occupancy.min(),
        'days_over_90pct': (daily_occupancy > 90).sum(),
        'avg_available_beds': total_beds - daily_census.mean()
    }

    # By day of week
    df['day_of_week'] = pd.to_datetime(df['census_date']).dt.day_name()
    by_dow = df.groupby('day_of_week')['patient_count'].mean()

    return metrics, daily_occupancy, by_dow
```

### Wait Time Analysis
```python
def analyze_wait_times(df, target_minutes=30):
    """
    Analyze patient wait times
    """

    # Calculate wait time
    df['wait_time'] = (df['service_start'] - df['arrival_time']).dt.total_seconds() / 60

    # Summary statistics
    stats = {
        'avg_wait': df['wait_time'].mean(),
        'median_wait': df['wait_time'].median(),
        'pct_within_target': (df['wait_time'] <= target_minutes).mean() * 100,
        'pct_over_60min': (df['wait_time'] > 60).mean() * 100
    }

    # By hour of day
    df['hour'] = pd.to_datetime(df['arrival_time']).dt.hour
    by_hour = df.groupby('hour')['wait_time'].mean()

    # By department
    by_dept = df.groupby('department')['wait_time'].agg(['mean', 'median', 'count'])

    return {
        'statistics': stats,
        'by_hour': by_hour,
        'by_department': by_dept
    }
```

## Population Health

### Risk Stratification
```python
def stratify_risk(df, risk_factors):
    """
    Stratify patient population by risk level
    """

    # Calculate risk score
    df['risk_score'] = 0

    for factor, weight in risk_factors.items():
        if factor in df.columns:
            df['risk_score'] += df[factor] * weight

    # Normalize to 0-100
    df['risk_score'] = (
        (df['risk_score'] - df['risk_score'].min()) /
        (df['risk_score'].max() - df['risk_score'].min()) * 100
    )

    # Assign risk tiers
    df['risk_tier'] = pd.cut(
        df['risk_score'],
        bins=[0, 25, 50, 75, 100],
        labels=['Low', 'Moderate', 'High', 'Very High']
    )

    # Summary
    summary = df.groupby('risk_tier').agg({
        'patient_id': 'count',
        'total_cost': 'mean',
        'ed_visits': 'mean',
        'hospitalizations': 'mean'
    })

    return df, summary

# Example risk factors
risk_factors = {
    'age_over_65': 2,
    'chronic_conditions': 3,
    'prior_hospitalizations': 4,
    'medication_count': 1,
    'social_risk_score': 2
}
```

### Care Gap Analysis
```python
def identify_care_gaps(df, quality_measures):
    """
    Identify patients with care gaps

    quality_measures: dict of {measure: compliance_column}
    """

    gaps = []

    for measure, compliance_col in quality_measures.items():
        non_compliant = df[df[compliance_col] == False]

        gaps.append({
            'measure': measure,
            'total_eligible': len(df),
            'non_compliant': len(non_compliant),
            'compliance_rate': df[compliance_col].mean() * 100,
            'gap_patients': non_compliant['patient_id'].tolist()
        })

    return pd.DataFrame(gaps).sort_values('compliance_rate')
```

## Quality Metrics

### Core Measures
```
COMMON QUALITY MEASURES:

PREVENTIVE CARE
- Mammography screening
- Colonoscopy screening
- Immunizations
- A1C testing for diabetics

CHRONIC DISEASE
- Blood pressure control
- Diabetes management
- CHF management
- COPD management

HOSPITAL QUALITY
- Mortality rates
- Readmission rates
- Patient safety indicators
- Hospital-acquired infections
```

### Quality Score Calculation
```python
def calculate_quality_score(df, measures, weights=None):
    """
    Calculate composite quality score
    """

    if weights is None:
        weights = {m: 1 for m in measures}

    # Calculate compliance for each measure
    scores = {}
    for measure in measures:
        scores[measure] = df[measure].mean() * 100

    # Weighted composite
    total_weight = sum(weights.values())
    composite = sum(scores[m] * weights[m] for m in measures) / total_weight

    return {
        'individual_scores': scores,
        'composite_score': composite
    }
```

## Financial Analysis

### Cost per Case
```python
def analyze_cost_per_case(df, drg_col='drg'):
    """
    Analyze cost per case by DRG
    """

    by_drg = df.groupby(drg_col).agg({
        'total_cost': ['mean', 'median', 'std', 'count'],
        'length_of_stay': 'mean',
        'revenue': 'mean'
    })

    # Profitability
    by_drg['margin'] = by_drg['revenue']['mean'] - by_drg['total_cost']['mean']
    by_drg['margin_pct'] = by_drg['margin'] / by_drg['revenue']['mean'] * 100

    return by_drg.sort_values(('total_cost', 'mean'), ascending=False)
```

## Reporting

### Healthcare Dashboard
```
HEALTHCARE ANALYTICS DASHBOARD
Period: [Date Range]

CLINICAL OUTCOMES
┌─────────────────────────────────────────────┐
│ 30-Day Readmission │ [X%]      │ [vs Target]│
│ Avg Length of Stay │ [X] days  │ [vs Bench] │
│ Mortality Rate     │ [X%]      │ [Trend]    │
│ Patient Safety     │ [Score]   │ [Trend]    │
└─────────────────────────────────────────────┘

OPERATIONAL METRICS
┌─────────────────────────────────────────────┐
│ Bed Occupancy      │ [X%]      │ [Trend]    │
│ ED Wait Time       │ [X] min   │ [vs Target]│
│ No-Show Rate       │ [X%]      │ [Trend]    │
│ Staff Utilization  │ [X%]      │ [Trend]    │
└─────────────────────────────────────────────┘

QUALITY MEASURES
┌────────────────────────────────────────────┐
│ Measure           │ Rate   │ Target │ Gap  │
├───────────────────┼────────┼────────┼──────┤
│ Diabetes A1C      │ [X%]   │ 85%    │ [X%] │
│ Mammography       │ [X%]   │ 80%    │ [X%] │
│ BP Control        │ [X%]   │ 75%    │ [X%] │
└────────────────────────────────────────────┘

POPULATION HEALTH
- High Risk Patients: [N] ([X%])
- Care Gaps Identified: [N]
- Patients Engaged: [N]

KEY INSIGHTS
1. [Clinical finding]
2. [Operational finding]
3. [Quality opportunity]

ACTIONS
1. [Recommendation]
2. [Recommendation]
```

## Checklist

### Healthcare Analytics Setup
```
□ Data governance in place (HIPAA)
□ Clinical data feeds established
□ Quality measure definitions aligned
□ Cost accounting methodology set
□ Benchmarks identified
```

### Regular Analysis
```
□ Monthly quality scorecard
□ Weekly operational metrics
□ Quarterly outcome review
□ Ongoing population stratification
□ Cost per case monitoring
```

Describe your healthcare data and question.

---
Downloaded from [Find Skill.ai](https://findskill.ai)