---
name: momentum-trading-strategy
version: 1.0.0
description: Momentum trading with RSI, MACD, and moving average analysis
author: FindSkill.ai
license: MIT
---

# Initialization

This skill provides a comprehensive momentum trading analysis framework. It includes Python scripts for momentum scanning and signal generation.

**Time to initialize**: ~3 minutes

## Directory Structure

```
momentum-trading-strategy/
├── SKILL.md                      # Main skill instructions
├── INIT.md                       # This initialization file
├── scripts/
│   ├── momentum_scanner.py       # Scan for momentum setups
│   └── rsi_calculator.py         # Calculate RSI values
└── references/
    └── momentum_signals.md       # Quick reference guide
```

## Dependencies

```bash
pip install pandas numpy yfinance
```

## Scripts

### scripts/rsi_calculator.py
**Purpose**: Calculate RSI and identify momentum signals
**Usage**: `python scripts/rsi_calculator.py --ticker AAPL --period 14`

```python
#!/usr/bin/env python3
"""
RSI Calculator and Momentum Signal Generator
Calculate RSI values and identify trading signals.
"""

import argparse
from typing import Tuple, List, Dict
import numpy as np

try:
    import yfinance as yf
    import pandas as pd
    YFINANCE_AVAILABLE = True
except ImportError:
    YFINANCE_AVAILABLE = False


def calculate_rsi(prices: List[float], period: int = 14) -> List[float]:
    """
    Calculate RSI from a list of prices.

    Args:
        prices: List of closing prices
        period: RSI calculation period (default 14)

    Returns:
        List of RSI values
    """
    if len(prices) < period + 1:
        raise ValueError(f"Need at least {period + 1} prices for RSI calculation")

    deltas = np.diff(prices)
    gains = np.where(deltas > 0, deltas, 0)
    losses = np.where(deltas < 0, -deltas, 0)

    avg_gain = np.zeros(len(deltas))
    avg_loss = np.zeros(len(deltas))

    # First average
    avg_gain[period-1] = np.mean(gains[:period])
    avg_loss[period-1] = np.mean(losses[:period])

    # Subsequent averages using smoothing
    for i in range(period, len(deltas)):
        avg_gain[i] = (avg_gain[i-1] * (period-1) + gains[i]) / period
        avg_loss[i] = (avg_loss[i-1] * (period-1) + losses[i]) / period

    rs = np.divide(avg_gain, avg_loss, out=np.zeros_like(avg_gain), where=avg_loss != 0)
    rsi = 100 - (100 / (1 + rs))

    # Pad beginning with NaN
    rsi[:period-1] = np.nan

    return rsi.tolist()


def analyze_momentum(rsi_values: List[float], current_price: float = None) -> Dict:
    """
    Analyze momentum based on RSI values.

    Args:
        rsi_values: List of RSI values
        current_price: Current price (optional)

    Returns:
        Dictionary with momentum analysis
    """
    current_rsi = rsi_values[-1]
    prev_rsi = rsi_values[-2] if len(rsi_values) > 1 else current_rsi

    # Determine momentum state
    if current_rsi >= 70:
        momentum_state = "STRONG BULLISH"
        momentum_score = 5
    elif current_rsi >= 50:
        momentum_state = "BULLISH"
        momentum_score = 4
    elif current_rsi >= 40:
        momentum_state = "NEUTRAL"
        momentum_score = 3
    elif current_rsi >= 30:
        momentum_state = "BEARISH"
        momentum_score = 2
    else:
        momentum_state = "STRONG BEARISH"
        momentum_score = 1

    # Detect RSI 50 crossover
    crossed_50_up = prev_rsi < 50 and current_rsi >= 50
    crossed_50_down = prev_rsi > 50 and current_rsi <= 50

    # Detect trend
    if len(rsi_values) >= 5:
        recent_trend = "RISING" if rsi_values[-1] > rsi_values[-5] else "FALLING"
    else:
        recent_trend = "UNKNOWN"

    # Generate signal
    if crossed_50_up:
        signal = "BUY SIGNAL - RSI crossed above 50"
        signal_strength = "STRONG"
    elif crossed_50_down:
        signal = "SELL SIGNAL - RSI crossed below 50"
        signal_strength = "STRONG"
    elif current_rsi > 70:
        signal = "OVERBOUGHT - Consider taking profits"
        signal_strength = "MODERATE"
    elif current_rsi < 30:
        signal = "OVERSOLD - Potential bounce"
        signal_strength = "MODERATE"
    else:
        signal = "NO CLEAR SIGNAL"
        signal_strength = "NONE"

    return {
        "current_rsi": round(current_rsi, 2),
        "previous_rsi": round(prev_rsi, 2),
        "momentum_state": momentum_state,
        "momentum_score": momentum_score,
        "recent_trend": recent_trend,
        "crossed_50_up": crossed_50_up,
        "crossed_50_down": crossed_50_down,
        "signal": signal,
        "signal_strength": signal_strength
    }


def fetch_and_analyze(ticker: str, period: int = 14, lookback: str = "3mo") -> Dict:
    """
    Fetch price data and perform momentum analysis.

    Args:
        ticker: Stock ticker symbol
        period: RSI period
        lookback: Historical data period

    Returns:
        Complete momentum analysis
    """
    if not YFINANCE_AVAILABLE:
        raise ImportError("yfinance not installed. Run: pip install yfinance")

    # Fetch data
    stock = yf.Ticker(ticker)
    hist = stock.history(period=lookback)

    if hist.empty:
        raise ValueError(f"No data found for {ticker}")

    prices = hist['Close'].tolist()
    current_price = prices[-1]

    # Calculate RSI
    rsi_values = calculate_rsi(prices, period)

    # Filter out NaN values for analysis
    valid_rsi = [r for r in rsi_values if not np.isnan(r)]

    # Analyze momentum
    analysis = analyze_momentum(valid_rsi, current_price)
    analysis["ticker"] = ticker
    analysis["current_price"] = round(current_price, 2)
    analysis["rsi_period"] = period

    return analysis


def print_analysis(analysis: Dict):
    """Print formatted momentum analysis."""
    print("\n" + "="*55)
    print(f"MOMENTUM ANALYSIS: {analysis.get('ticker', 'N/A')}")
    print("="*55)

    print(f"\n📊 PRICE & RSI")
    print(f"   Current Price:    ${analysis.get('current_price', 'N/A')}")
    print(f"   Current RSI({analysis.get('rsi_period', 14)}):  {analysis['current_rsi']}")
    print(f"   Previous RSI:     {analysis['previous_rsi']}")

    print(f"\n📈 MOMENTUM STATUS")
    print(f"   State:            {analysis['momentum_state']}")
    print(f"   Score:            {analysis['momentum_score']}/5")
    print(f"   Trend:            {analysis['recent_trend']}")

    print(f"\n🎯 SIGNAL")
    print(f"   {analysis['signal']}")
    print(f"   Strength:         {analysis['signal_strength']}")

    if analysis['crossed_50_up']:
        print(f"\n   ✅ BULLISH: RSI just crossed above 50!")
    elif analysis['crossed_50_down']:
        print(f"\n   ⚠️  BEARISH: RSI just crossed below 50!")

    print("\n" + "="*55)

    # Trading recommendation
    print("\n💡 RECOMMENDATION")
    if analysis['momentum_score'] >= 4 and analysis['recent_trend'] == 'RISING':
        print("   Strong momentum - consider entry on pullback")
    elif analysis['momentum_score'] <= 2:
        print("   Weak momentum - avoid longs, consider shorts")
    else:
        print("   Mixed signals - wait for clearer direction")

    print("\n")


def main():
    parser = argparse.ArgumentParser(description="RSI Momentum Calculator")

    parser.add_argument("--ticker", type=str, help="Stock ticker symbol")
    parser.add_argument("--period", type=int, default=14, help="RSI period (default: 14)")
    parser.add_argument("--prices", type=str, help="Comma-separated prices (alternative to ticker)")

    args = parser.parse_args()

    if args.ticker:
        analysis = fetch_and_analyze(args.ticker, args.period)
        print_analysis(analysis)
    elif args.prices:
        prices = [float(p.strip()) for p in args.prices.split(",")]
        rsi_values = calculate_rsi(prices, args.period)
        valid_rsi = [r for r in rsi_values if not np.isnan(r)]
        analysis = analyze_momentum(valid_rsi)
        analysis["current_price"] = prices[-1]
        analysis["rsi_period"] = args.period
        analysis["ticker"] = "CUSTOM"
        print_analysis(analysis)
    else:
        print("Please provide either --ticker or --prices")
        parser.print_help()


if __name__ == "__main__":
    main()
```

### scripts/momentum_scanner.py
**Purpose**: Scan multiple stocks for momentum setups
**Usage**: `python scripts/momentum_scanner.py --tickers AAPL,MSFT,GOOGL,NVDA`

```python
#!/usr/bin/env python3
"""
Momentum Scanner
Scan multiple stocks for momentum trading setups.
"""

import argparse
from typing import List, Dict
import numpy as np

try:
    import yfinance as yf
    import pandas as pd
    YFINANCE_AVAILABLE = True
except ImportError:
    YFINANCE_AVAILABLE = False


def calculate_rsi(prices: List[float], period: int = 14) -> float:
    """Calculate current RSI value."""
    if len(prices) < period + 1:
        return np.nan

    deltas = np.diff(prices)
    gains = np.where(deltas > 0, deltas, 0)
    losses = np.where(deltas < 0, -deltas, 0)

    avg_gain = np.mean(gains[-period:])
    avg_loss = np.mean(losses[-period:])

    if avg_loss == 0:
        return 100.0

    rs = avg_gain / avg_loss
    return 100 - (100 / (1 + rs))


def calculate_macd(prices: List[float]) -> Dict:
    """Calculate MACD values."""
    prices = np.array(prices)

    # EMAs
    ema12 = pd.Series(prices).ewm(span=12).mean().iloc[-1]
    ema26 = pd.Series(prices).ewm(span=26).mean().iloc[-1]

    macd_line = ema12 - ema26

    # Signal line (9 EMA of MACD)
    macd_series = pd.Series(prices).ewm(span=12).mean() - pd.Series(prices).ewm(span=26).mean()
    signal_line = macd_series.ewm(span=9).mean().iloc[-1]

    histogram = macd_line - signal_line

    return {
        "macd": round(macd_line, 4),
        "signal": round(signal_line, 4),
        "histogram": round(histogram, 4),
        "bullish": macd_line > signal_line
    }


def scan_stock(ticker: str, lookback: str = "3mo") -> Dict:
    """Scan a single stock for momentum signals."""
    if not YFINANCE_AVAILABLE:
        raise ImportError("yfinance required. Run: pip install yfinance pandas")

    try:
        stock = yf.Ticker(ticker)
        hist = stock.history(period=lookback)

        if hist.empty or len(hist) < 30:
            return {"ticker": ticker, "error": "Insufficient data"}

        prices = hist['Close'].tolist()
        volumes = hist['Volume'].tolist()

        current_price = prices[-1]
        prev_price = prices[-2]

        # Calculate indicators
        rsi = calculate_rsi(prices)
        macd_data = calculate_macd(prices)

        # Calculate MAs
        ema10 = pd.Series(prices).ewm(span=10).mean().iloc[-1]
        ema20 = pd.Series(prices).ewm(span=20).mean().iloc[-1]
        sma50 = np.mean(prices[-50:]) if len(prices) >= 50 else np.nan

        # Volume analysis
        avg_volume = np.mean(volumes[-20:])
        current_volume = volumes[-1]
        volume_ratio = current_volume / avg_volume if avg_volume > 0 else 1

        # Price change
        price_change = ((current_price - prev_price) / prev_price) * 100

        # Momentum score (0-10)
        score = 0
        reasons = []

        # RSI scoring
        if 50 < rsi < 70:
            score += 2
            reasons.append("RSI bullish (50-70)")
        elif rsi > 70:
            score += 1
            reasons.append("RSI overbought")
        elif rsi < 30:
            score += 1
            reasons.append("RSI oversold (potential bounce)")

        # MACD scoring
        if macd_data["bullish"]:
            score += 2
            reasons.append("MACD bullish crossover")
        if macd_data["histogram"] > 0:
            score += 1
            reasons.append("MACD histogram positive")

        # MA scoring
        if current_price > ema10 > ema20:
            score += 2
            reasons.append("Price > 10 EMA > 20 EMA")
        if not np.isnan(sma50) and current_price > sma50:
            score += 1
            reasons.append("Above 50 SMA")

        # Volume scoring
        if volume_ratio > 1.5:
            score += 1
            reasons.append("High volume")

        # Determine signal
        if score >= 7:
            signal = "STRONG BUY"
        elif score >= 5:
            signal = "BUY"
        elif score >= 3:
            signal = "NEUTRAL"
        elif score >= 1:
            signal = "WEAK"
        else:
            signal = "AVOID"

        return {
            "ticker": ticker,
            "price": round(current_price, 2),
            "change": round(price_change, 2),
            "rsi": round(rsi, 1),
            "macd_bullish": macd_data["bullish"],
            "histogram": macd_data["histogram"],
            "above_10ema": current_price > ema10,
            "above_20ema": current_price > ema20,
            "volume_ratio": round(volume_ratio, 2),
            "momentum_score": score,
            "signal": signal,
            "reasons": reasons
        }

    except Exception as e:
        return {"ticker": ticker, "error": str(e)}


def scan_multiple(tickers: List[str]) -> List[Dict]:
    """Scan multiple stocks and return sorted by momentum score."""
    results = []
    for ticker in tickers:
        result = scan_stock(ticker)
        if "error" not in result:
            results.append(result)
        else:
            print(f"⚠️  Skipping {ticker}: {result['error']}")

    # Sort by momentum score descending
    results.sort(key=lambda x: x["momentum_score"], reverse=True)
    return results


def print_scan_results(results: List[Dict]):
    """Print formatted scan results."""
    print("\n" + "="*80)
    print("MOMENTUM SCANNER RESULTS")
    print("="*80)

    print(f"\n{'Ticker':<8} {'Price':>10} {'Chg%':>8} {'RSI':>6} {'MACD':>8} {'Vol':>6} {'Score':>6} {'Signal':<12}")
    print("-"*80)

    for r in results:
        macd_str = "✅" if r["macd_bullish"] else "❌"
        print(f"{r['ticker']:<8} ${r['price']:>9.2f} {r['change']:>7.2f}% {r['rsi']:>5.1f} {macd_str:>8} {r['volume_ratio']:>5.1f}x {r['momentum_score']:>5}/10 {r['signal']:<12}")

    print("="*80)

    # Show top picks
    strong_buys = [r for r in results if r["signal"] in ["STRONG BUY", "BUY"]]
    if strong_buys:
        print("\n🎯 TOP MOMENTUM PICKS:")
        for r in strong_buys[:5]:
            print(f"\n   {r['ticker']} (Score: {r['momentum_score']}/10)")
            for reason in r["reasons"][:3]:
                print(f"      • {reason}")

    print("\n")


def main():
    parser = argparse.ArgumentParser(description="Momentum Scanner")

    parser.add_argument(
        "--tickers",
        type=str,
        default="AAPL,MSFT,GOOGL,AMZN,NVDA,META,TSLA,AMD,NFLX,CRM",
        help="Comma-separated ticker symbols"
    )

    args = parser.parse_args()

    tickers = [t.strip().upper() for t in args.tickers.split(",")]

    print(f"\n🔍 Scanning {len(tickers)} stocks for momentum...")

    results = scan_multiple(tickers)
    print_scan_results(results)


if __name__ == "__main__":
    main()
```

## References

### references/momentum_signals.md

```markdown
# Momentum Signal Quick Reference

## RSI Momentum Signals

| Signal | Condition | Interpretation |
|--------|-----------|----------------|
| Bullish Crossover | RSI crosses above 50 | Momentum turning bullish |
| Bearish Crossover | RSI crosses below 50 | Momentum turning bearish |
| Strong Trend | RSI 60-80 | Healthy bullish momentum |
| Overbought | RSI > 70 | Consider trailing stops |
| Oversold | RSI < 30 | Potential bounce |
| Bullish Divergence | Price lower low, RSI higher low | Reversal signal |
| Bearish Divergence | Price higher high, RSI lower high | Reversal signal |

## MACD Momentum Signals

| Signal | Condition | Action |
|--------|-----------|--------|
| Bullish Crossover | MACD > Signal | Enter long |
| Bearish Crossover | MACD < Signal | Exit long / enter short |
| Histogram Expansion | Bars increasing | Hold position |
| Histogram Contraction | Bars decreasing | Tighten stops |
| Zero Line Cross Up | MACD crosses above 0 | Trend confirmation |

## Moving Average Momentum

### Bullish Alignment
```
Price > 10 EMA > 20 EMA > 50 SMA
All MAs sloping upward
```

### Bearish Alignment
```
Price < 10 EMA < 20 EMA < 50 SMA
All MAs sloping downward
```

## Triple Confirmation Entry

Enter when ALL align:
1. ✅ RSI above 50 and rising
2. ✅ MACD above signal, histogram positive
3. ✅ Price above 20 EMA, 20 EMA above 50 EMA

## Exit Signals

**Exit when:**
- RSI shows bearish divergence
- MACD histogram contracts significantly
- Price closes below 20 EMA
- Volume spikes on down day
```

## File Extraction Rules

1. **Scripts**: Extract code after `### scripts/{filename}` → save to `scripts/`
2. **References**: Extract code after `### references/{filename}` → save to `references/`
3. Create directories if needed
4. Make scripts executable: `chmod +x scripts/*.py`

## Post-Init Steps

1. Create directories:
   ```bash
   mkdir -p scripts references
   ```

2. Extract and save files from code blocks above

3. Install dependencies:
   ```bash
   pip install pandas numpy yfinance
   ```

4. Test RSI calculator:
   ```bash
   python scripts/rsi_calculator.py --ticker AAPL
   ```

5. Test momentum scanner:
   ```bash
   python scripts/momentum_scanner.py --tickers AAPL,MSFT,GOOGL
   ```

## Compatibility

Tested with: Claude, ChatGPT, Gemini, Copilot

The main skill (SKILL.md) works without initialization. Scripts provide enhanced functionality but are optional.

---
Downloaded from [Find Skill.ai](https://findskill.ai)
