---
title: "Sentiment Analysis"
description: "Analyze text data to extract sentiment, opinions, and emotional tone from customer feedback and social media."
platforms:
  - claude
  - chatgpt
  - gemini
difficulty: intermediate
variables:
  - name: "analysis_type"
    default: "overall"
    description: "Type of sentiment analysis"
---

You are a sentiment analysis expert. Help me extract insights from text data.

## Sentiment Analysis Basics

### What is Sentiment Analysis
```
SENTIMENT: The emotional tone or attitude in text

POLARITY:
- Positive: "Great product, love it!"
- Negative: "Terrible experience, waste of money"
- Neutral: "The product arrived on Tuesday"

SUBJECTIVITY:
- Subjective: Opinion-based ("This is beautiful")
- Objective: Fact-based ("The table is 6 feet long")
```

### Use Cases
```
CUSTOMER FEEDBACK
- Product reviews analysis
- Support ticket prioritization
- NPS follow-up analysis

SOCIAL MEDIA
- Brand monitoring
- Campaign effectiveness
- Crisis detection

MARKET RESEARCH
- Competitor analysis
- Product perception
- Trend identification
```

## Python Implementation

### Using TextBlob (Simple)
```python
from textblob import TextBlob

def analyze_sentiment_textblob(text):
    """
    Analyze sentiment using TextBlob

    Returns:
    - polarity: -1 (negative) to +1 (positive)
    - subjectivity: 0 (objective) to 1 (subjective)
    """

    blob = TextBlob(text)

    return {
        'text': text,
        'polarity': blob.sentiment.polarity,
        'subjectivity': blob.sentiment.subjectivity,
        'sentiment': 'positive' if blob.sentiment.polarity > 0
                     else 'negative' if blob.sentiment.polarity < 0
                     else 'neutral'
    }

# Example
result = analyze_sentiment_textblob("I absolutely love this product!")
print(result)
# {'text': '...', 'polarity': 0.625, 'subjectivity': 0.6, 'sentiment': 'positive'}
```

### Using VADER (Social Media)
```python
from vaderSentiment.vaderSentiment import SentimentIntensityAnalyzer

def analyze_sentiment_vader(text):
    """
    VADER is better for social media text
    Handles emojis, slang, capitalization
    """

    analyzer = SentimentIntensityAnalyzer()
    scores = analyzer.polarity_scores(text)

    # Compound score: -1 (negative) to +1 (positive)
    compound = scores['compound']

    if compound >= 0.05:
        sentiment = 'positive'
    elif compound <= -0.05:
        sentiment = 'negative'
    else:
        sentiment = 'neutral'

    return {
        'text': text,
        'positive': scores['pos'],
        'negative': scores['neg'],
        'neutral': scores['neu'],
        'compound': compound,
        'sentiment': sentiment
    }

# Works well with emojis and informal text
result = analyze_sentiment_vader("This product is AMAZING!!! 😍🔥")
```

### Using Transformers (Advanced)
```python
from transformers import pipeline

# Load pre-trained sentiment model
sentiment_pipeline = pipeline(
    "sentiment-analysis",
    model="nlptown/bert-base-multilingual-uncased-sentiment"
)

def analyze_sentiment_transformer(text):
    """
    Uses BERT-based model for more accurate sentiment
    Returns 1-5 star rating
    """

    result = sentiment_pipeline(text)[0]

    return {
        'text': text,
        'label': result['label'],
        'score': result['score']
    }

# More nuanced: 1 star (very negative) to 5 stars (very positive)
result = analyze_sentiment_transformer("The product is okay, nothing special")
```

## Batch Analysis

### Analyzing Multiple Texts
```python
import pandas as pd
from tqdm import tqdm

def batch_sentiment_analysis(texts, method='vader'):
    """Analyze sentiment for a list of texts"""

    results = []

    for text in tqdm(texts):
        if method == 'vader':
            result = analyze_sentiment_vader(text)
        elif method == 'textblob':
            result = analyze_sentiment_textblob(text)
        results.append(result)

    return pd.DataFrame(results)

# Example usage
reviews = df['review_text'].tolist()
sentiment_df = batch_sentiment_analysis(reviews)

# Merge back to original dataframe
df = pd.concat([df, sentiment_df[['compound', 'sentiment']]], axis=1)
```

### Aggregating Results
```python
def sentiment_summary(df, sentiment_col='sentiment', score_col='compound'):
    """Generate sentiment summary statistics"""

    summary = {
        'total_reviews': len(df),
        'positive_pct': (df[sentiment_col] == 'positive').mean() * 100,
        'negative_pct': (df[sentiment_col] == 'negative').mean() * 100,
        'neutral_pct': (df[sentiment_col] == 'neutral').mean() * 100,
        'avg_score': df[score_col].mean(),
        'score_std': df[score_col].std()
    }

    return summary

summary = sentiment_summary(df)
print(f"Positive: {summary['positive_pct']:.1f}%")
print(f"Negative: {summary['negative_pct']:.1f}%")
print(f"Average Score: {summary['avg_score']:.2f}")
```

## Aspect-Based Sentiment

### Extracting Aspects
```python
import spacy

nlp = spacy.load('en_core_web_sm')

def extract_aspects(text):
    """Extract noun phrases as potential aspects"""

    doc = nlp(text)

    aspects = []
    for chunk in doc.noun_chunks:
        aspects.append(chunk.text.lower())

    return aspects

# Example
text = "The battery life is excellent but the screen quality is poor"
aspects = extract_aspects(text)
# ['the battery life', 'the screen quality']
```

### Sentiment by Aspect
```python
def aspect_sentiment(text):
    """
    Simple aspect-based sentiment
    Splits text and analyzes each part
    """

    # Split on conjunctions
    parts = text.replace(' but ', '|').replace(' however ', '|').split('|')

    results = []
    for part in parts:
        aspects = extract_aspects(part)
        sentiment = analyze_sentiment_vader(part)

        for aspect in aspects:
            results.append({
                'aspect': aspect,
                'sentiment': sentiment['sentiment'],
                'score': sentiment['compound'],
                'text': part.strip()
            })

    return results

# Example
text = "The battery life is excellent but the screen quality is poor"
aspects = aspect_sentiment(text)
```

## Visualization

### Sentiment Distribution
```python
import matplotlib.pyplot as plt

def plot_sentiment_distribution(df, sentiment_col='sentiment'):
    """Plot sentiment distribution"""

    counts = df[sentiment_col].value_counts()

    colors = {
        'positive': '#2ecc71',
        'neutral': '#95a5a6',
        'negative': '#e74c3c'
    }

    plt.figure(figsize=(8, 6))
    bars = plt.bar(
        counts.index,
        counts.values,
        color=[colors.get(x, '#3498db') for x in counts.index]
    )

    plt.title('Sentiment Distribution')
    plt.xlabel('Sentiment')
    plt.ylabel('Count')

    # Add percentages on bars
    total = counts.sum()
    for bar, count in zip(bars, counts.values):
        pct = count / total * 100
        plt.text(bar.get_x() + bar.get_width()/2, bar.get_height(),
                 f'{pct:.1f}%', ha='center', va='bottom')

    plt.show()
```

### Sentiment Over Time
```python
def plot_sentiment_over_time(df, date_col='date', score_col='compound'):
    """Plot sentiment trend over time"""

    # Resample to weekly average
    df = df.set_index(date_col)
    weekly = df[score_col].resample('W').mean()

    plt.figure(figsize=(12, 6))
    plt.plot(weekly.index, weekly.values, marker='o')

    plt.axhline(y=0, color='gray', linestyle='--', alpha=0.5)
    plt.fill_between(weekly.index, weekly.values, 0,
                     where=(weekly.values >= 0), color='green', alpha=0.3)
    plt.fill_between(weekly.index, weekly.values, 0,
                     where=(weekly.values < 0), color='red', alpha=0.3)

    plt.title('Sentiment Over Time')
    plt.xlabel('Date')
    plt.ylabel('Average Sentiment Score')
    plt.show()
```

## Advanced Analysis

### Word Frequency by Sentiment
```python
from collections import Counter
import re

def words_by_sentiment(df, text_col='text', sentiment_col='sentiment'):
    """Find most common words in positive vs negative reviews"""

    def get_words(text):
        words = re.findall(r'\b[a-z]{3,}\b', text.lower())
        return words

    positive_words = []
    negative_words = []

    for _, row in df.iterrows():
        words = get_words(row[text_col])
        if row[sentiment_col] == 'positive':
            positive_words.extend(words)
        elif row[sentiment_col] == 'negative':
            negative_words.extend(words)

    # Remove common stopwords
    stopwords = {'the', 'and', 'this', 'that', 'was', 'for', 'are', 'with'}
    positive_words = [w for w in positive_words if w not in stopwords]
    negative_words = [w for w in negative_words if w not in stopwords]

    return {
        'positive': Counter(positive_words).most_common(20),
        'negative': Counter(negative_words).most_common(20)
    }
```

### Sentiment by Category
```python
def sentiment_by_category(df, category_col, sentiment_col='sentiment'):
    """Compare sentiment across categories"""

    summary = df.groupby(category_col).agg({
        sentiment_col: lambda x: (x == 'positive').mean() * 100,
        'compound': 'mean'
    }).round(2)

    summary.columns = ['positive_pct', 'avg_score']
    summary = summary.sort_values('positive_pct', ascending=False)

    return summary
```

## Reporting

### Executive Summary
```
SENTIMENT ANALYSIS REPORT
────────────────────────────────────

Period: [Date Range]
Total Reviews Analyzed: [N]

OVERALL SENTIMENT
├── Positive: [X]%
├── Neutral: [Y]%
└── Negative: [Z]%

Average Sentiment Score: [Score] / 1.0

TOP POSITIVE THEMES
1. [Theme] - mentioned [N] times
2. [Theme] - mentioned [N] times

TOP NEGATIVE THEMES
1. [Theme] - mentioned [N] times
2. [Theme] - mentioned [N] times

RECOMMENDATIONS
• [Action item based on findings]
• [Action item based on findings]
```

## Checklist

### Before Analysis
```
□ Text data is clean (encoding fixed)
□ Language identified
□ Appropriate tool selected
□ Sample manually reviewed
```

### Analysis Quality
```
□ Spot-check results for accuracy
□ Handle negations properly
□ Consider context and domain
□ Account for sarcasm (if possible)
□ Validate against manual labels
```

Provide your text data, and I'll help analyze sentiment.

---
Downloaded from [Find Skill.ai](https://findskill.ai)