# Accessible Color Palette Generator - Skill Instructions

## Purpose

This skill enables AI assistants to generate WCAG-compliant color palettes with guaranteed contrast ratios for accessible UI design. It creates complete color systems that work across light and dark themes while ensuring text readability and colorblind accessibility.

## Core Functionality

### 1. Color Scale Generation

Generate 11-step color scales (50-950) for each semantic role:
- **50**: Lightest tint (95% lightness) - subtle backgrounds
- **100**: Very light (90%) - hover states on light backgrounds
- **200**: Light (80%) - disabled states
- **300**: Light medium (70%) - borders, dividers
- **400**: Medium light (60%) - placeholder text
- **500**: Base color - primary interactive elements
- **600**: Medium dark (40%) - hover states on base
- **700**: Dark (30%) - high contrast text
- **800**: Darker (20%) - headings, emphasis
- **900**: Very dark (15%) - maximum contrast
- **950**: Nearly black (10%) - dark mode backgrounds

### 2. Semantic Color Roles

Generate the following color families:

#### Primary
- Derived from user's brand color input
- Full 50-950 scale
- Used for main CTAs, links, brand elements

#### Secondary
- Complementary or analogous to primary
- Full 50-950 scale
- Used for secondary actions, accents

#### Success (Positive)
- Green family (hue 120-150°)
- Communicates completion, success, positive states
- Must be distinguishable from primary and error

#### Warning (Caution)
- Amber/Orange family (hue 35-45°)
- Communicates warnings, pending states
- Must contrast with both success and error

#### Error (Danger)
- Red family (hue 0-10° or 350-360°)
- Communicates errors, destructive actions
- Highest priority for colorblind-safe verification

#### Info
- Blue family (hue 200-220°)
- Communicates neutral information, help text
- Should not conflict with primary if primary is blue

#### Neutral (Gray Scale)
- Achromatic scale (0° hue, 0-5% saturation)
- Full 50-950 scale
- Used for text, borders, backgrounds, shadows

### 3. Accessibility Calculations

For every generated color, calculate:

#### WCAG 2.1 Contrast Ratios
Use formula: `(L1 + 0.05) / (L2 + 0.05)` where L1 > L2

Where L (relative luminance) =
- For RGB values 0-255, normalize to 0-1
- For each channel: if ≤ 0.03928, divide by 12.92, else ((value + 0.055) / 1.055) ^ 2.4
- L = 0.2126 × R + 0.7152 × G + 0.0722 × B

**WCAG AA Requirements:**
- Normal text (< 18pt): 4.5:1 minimum
- Large text (≥ 18pt or 14pt bold): 3:1 minimum
- UI components and graphics: 3:1 minimum

**WCAG AAA Requirements:**
- Normal text: 7:1 minimum
- Large text: 4.5:1 minimum

#### APCA Contrast
Use modern APCA (Accessible Perceptual Contrast Algorithm):
- More accurate perceptual contrast
- Polarity-aware (light-on-dark vs dark-on-light)
- Target: Lc 60+ for body text, Lc 75+ for small text
- Reference: https://github.com/Myndex/SAPC-APCA

#### Text Color Pairing
For each background color:
1. Test both white (#FFFFFF) and dark (#0A0A0A or neutral-950) text
2. Calculate contrast ratio for both options
3. Select the option with higher contrast
4. If neither meets WCAG level, adjust background or flag as non-compliant
5. Provide the "on-{color}" token (e.g., on-primary-500)

### 4. Colorblind-Safe Verification

When enabled, simulate the following types of color blindness:

#### Protanopia (Red-blind, ~1% of males)
- Reduced sensitivity to red light
- Confuse red/green, red/black
- Critical: Ensure error (red) and success (green) are distinguishable

#### Deuteranopia (Green-blind, ~1% of males)
- Reduced sensitivity to green light
- Most common form of color blindness
- Confuse red/green, green/brown

#### Tritanopia (Blue-blind, ~0.01% of population)
- Reduced sensitivity to blue light
- Confuse blue/green, yellow/violet

**Verification Process:**
1. Convert each semantic color to simulated colorblind perception
2. Calculate perceptual difference (ΔE) between simulated colors
3. Ensure ΔE ≥ 30 for critical pairs (success/error, primary/secondary)
4. Flag problematic combinations
5. Suggest adjustments (increase saturation difference, add lightness contrast)

### 5. Dark Mode Generation

Create dark mode variants by:

1. **Invert the lightness scale:**
   - Light mode 50 → Dark mode 950
   - Light mode 100 → Dark mode 900
   - ... and so on
   - Light mode 950 → Dark mode 50

2. **Adjust saturation:**
   - Reduce by 5-10% for more comfortable viewing
   - Higher lightness colors (now darker) may need slightly higher saturation

3. **Re-validate contrast:**
   - Dark backgrounds require same contrast ratios
   - Text on dark mode colors must meet WCAG standards
   - Adjust individual values if needed

4. **Semantic consistency:**
   - Success still looks "successful" (greenish)
   - Error still looks "dangerous" (reddish)
   - Maintain color meaning across themes

### 6. Style Preferences

Adjust generation based on style:

#### Vibrant
- High saturation (70-90%)
- Bolder colors
- Higher chroma differences between steps

#### Muted
- Lower saturation (30-50%)
- Softer colors
- Subtle step differences

#### Earthy
- Warm hues, slight yellow/brown shift
- Medium saturation (40-60%)
- Natural, organic feel

#### Pastel
- High lightness for lighter shades
- Medium-low saturation (40-60%)
- Soft, gentle appearance

#### Bold
- Maximum saturation where accessible
- High contrast between steps
- Dramatic, energetic

#### Minimal
- Low saturation (10-30%)
- Near-grayscale with subtle hue
- Clean, modern aesthetic

## Input Processing

### Required Input
- **Primary color**: Accept hex (#3B82F6), RGB (rgb(59, 130, 246)), or HSL (hsl(217, 91%, 60%))

### Optional Inputs (with defaults)
- **WCAG level**: "AA" (default) or "AAA"
- **Style preference**: "vibrant" (default), "muted", "earthy", "pastel", "bold", "minimal"
- **Colorblind safe**: true (default) or false
- **Output format**: "all" (default), "css", "tailwind", "scss", "json"

### Parse Primary Color
```javascript
// Example parsing logic
function parseColor(input) {
  // Hex
  if (input.startsWith('#')) {
    return hexToHSL(input);
  }
  // RGB
  if (input.startsWith('rgb')) {
    return rgbToHSL(input);
  }
  // HSL
  if (input.startsWith('hsl')) {
    return parseHSL(input);
  }
  throw new Error('Invalid color format');
}
```

## Output Formats

### 1. CSS Custom Properties

```css
:root {
  /* Primary Scale */
  --color-primary-50: hsl(217, 91%, 95%);
  --color-primary-100: hsl(217, 91%, 90%);
  --color-primary-200: hsl(217, 91%, 80%);
  --color-primary-300: hsl(217, 91%, 70%);
  --color-primary-400: hsl(217, 91%, 60%);
  --color-primary-500: hsl(217, 91%, 50%);
  --color-primary-600: hsl(217, 91%, 40%);
  --color-primary-700: hsl(217, 91%, 30%);
  --color-primary-800: hsl(217, 91%, 20%);
  --color-primary-900: hsl(217, 91%, 15%);
  --color-primary-950: hsl(217, 91%, 10%);

  /* Text colors for primary backgrounds */
  --color-on-primary-50: hsl(0, 0%, 10%);  /* Dark text */
  --color-on-primary-500: hsl(0, 0%, 100%); /* White text */
  --color-on-primary-700: hsl(0, 0%, 100%); /* White text */

  /* Success Scale */
  --color-success-50: hsl(142, 71%, 95%);
  --color-success-500: hsl(142, 71%, 45%);
  --color-success-on-500: hsl(0, 0%, 100%);

  /* Warning Scale */
  --color-warning-50: hsl(38, 92%, 95%);
  --color-warning-500: hsl(38, 92%, 50%);
  --color-warning-on-500: hsl(0, 0%, 10%);

  /* Error Scale */
  --color-error-50: hsl(0, 84%, 95%);
  --color-error-500: hsl(0, 84%, 50%);
  --color-error-on-500: hsl(0, 0%, 100%);

  /* Info Scale */
  --color-info-50: hsl(200, 94%, 95%);
  --color-info-500: hsl(200, 94%, 45%);
  --color-info-on-500: hsl(0, 0%, 100%);

  /* Neutral Scale */
  --color-neutral-50: hsl(0, 0%, 98%);
  --color-neutral-100: hsl(0, 0%, 96%);
  --color-neutral-200: hsl(0, 0%, 90%);
  --color-neutral-300: hsl(0, 0%, 80%);
  --color-neutral-400: hsl(0, 0%, 60%);
  --color-neutral-500: hsl(0, 0%, 50%);
  --color-neutral-600: hsl(0, 0%, 40%);
  --color-neutral-700: hsl(0, 0%, 30%);
  --color-neutral-800: hsl(0, 0%, 20%);
  --color-neutral-900: hsl(0, 0%, 15%);
  --color-neutral-950: hsl(0, 0%, 10%);
}

/* Dark Mode Overrides */
[data-theme="dark"] {
  --color-primary-50: hsl(217, 86%, 10%);
  --color-primary-100: hsl(217, 86%, 15%);
  --color-primary-500: hsl(217, 86%, 60%);
  /* ... inverted scale */
}
```

### 2. Tailwind CSS Config

```javascript
// tailwind.config.js
module.exports = {
  theme: {
    extend: {
      colors: {
        primary: {
          50: '#EFF6FF',
          100: '#DBEAFE',
          200: '#BFDBFE',
          300: '#93C5FD',
          400: '#60A5FA',
          500: '#3B82F6',
          600: '#2563EB',
          700: '#1D4ED8',
          800: '#1E40AF',
          900: '#1E3A8A',
          950: '#172554',
        },
        success: {
          50: '#F0FDF4',
          500: '#22C55E',
          // ... full scale
        },
        warning: {
          50: '#FFFBEB',
          500: '#F59E0B',
        },
        error: {
          50: '#FEF2F2',
          500: '#EF4444',
        },
        info: {
          50: '#EFF6FF',
          500: '#3B82F6',
        },
        neutral: {
          50: '#FAFAFA',
          500: '#737373',
          950: '#0A0A0A',
        },
      },
    },
  },
  plugins: [],
};
```

### 3. SCSS Variables

```scss
// _colors.scss

// Primary
$primary-50: hsl(217, 91%, 95%);
$primary-500: hsl(217, 91%, 50%);
$primary-on-500: hsl(0, 0%, 100%);

// Success
$success-50: hsl(142, 71%, 95%);
$success-500: hsl(142, 71%, 45%);

// Map for programmatic access
$colors: (
  'primary': (
    50: $primary-50,
    100: $primary-100,
    500: $primary-500,
    // ...
  ),
  'success': (
    // ...
  ),
);

// Helper function
@function color($name, $variant: 500) {
  @return map-get(map-get($colors, $name), $variant);
}

// Usage: background-color: color('primary', 500);
```

### 4. JSON Design Tokens (Figma)

```json
{
  "color": {
    "primary": {
      "50": {
        "value": "hsl(217, 91%, 95%)",
        "type": "color",
        "description": "Lightest primary tint for subtle backgrounds"
      },
      "500": {
        "value": "hsl(217, 91%, 50%)",
        "type": "color",
        "description": "Base primary color for interactive elements"
      },
      "on-500": {
        "value": "hsl(0, 0%, 100%)",
        "type": "color",
        "description": "Text color for primary-500 backgrounds (4.8:1 contrast)"
      }
    },
    "success": {
      "500": {
        "value": "hsl(142, 71%, 45%)",
        "type": "color",
        "description": "Success state color"
      }
    }
  },
  "meta": {
    "wcag_level": "AA",
    "colorblind_safe": true,
    "generated_date": "2025-08-15",
    "base_color": "#3B82F6"
  }
}
```

## Accessibility Report Format

For each color in the palette, provide:

```
PRIMARY-500: hsl(217, 91%, 50%) | #3B82F6
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
Contrast vs White (#FFFFFF): 4.8:1
  ✓ WCAG AA Normal Text (4.5:1)
  ✗ WCAG AAA Normal Text (7:1)
  ✓ WCAG AA Large Text (3:1)
  ✓ WCAG AAA Large Text (4.5:1)
  APCA: Lc 68.5 (readable for body text)

Contrast vs Black (#000000): 4.4:1
  ✗ WCAG AA Normal Text (4.5:1)
  ✓ WCAG AA Large Text (3:1)

Contrast vs Neutral-50 (#FAFAFA): 4.6:1
  ✓ WCAG AA Normal Text

Recommended Text Color: White (#FFFFFF)
  Contrast: 4.8:1 ✓ AA

Colorblind Safety:
  ✓ Protanopia: Remains distinct from success/error
  ✓ Deuteranopia: Remains distinct
  ✓ Tritanopia: Remains distinct

Usage Recommendations:
  ✓ Safe for primary buttons with white text
  ✓ Safe for large headings on white background
  ✗ Avoid for body text on white (use primary-700 instead)
```

## Usage Examples to Provide

Always include 3-5 practical examples:

### Example 1: Button Combinations
```html
<!-- Primary Button -->
<button style="background: var(--color-primary-600); color: var(--color-on-primary-600);">
  Click Me (Contrast: 5.2:1 ✓)
</button>

<!-- Success Button -->
<button style="background: var(--color-success-600); color: var(--color-on-success-600);">
  Save (Contrast: 4.7:1 ✓)
</button>
```

### Example 2: Text Hierarchy
```html
<!-- High contrast heading -->
<h1 style="color: var(--color-neutral-900);">
  Heading (Contrast vs white bg: 17.5:1 ✓ AAA)
</h1>

<!-- Body text -->
<p style="color: var(--color-neutral-700);">
  Body text (Contrast: 7.2:1 ✓ AAA)
</p>

<!-- Subtle text -->
<span style="color: var(--color-neutral-600);">
  Caption (Contrast: 4.8:1 ✓ AA)
</span>
```

### Example 3: Alert/Status Messages
```html
<!-- Error message -->
<div style="background: var(--color-error-50); color: var(--color-error-800);">
  Error: Please try again (Contrast: 8.1:1 ✓ AAA)
</div>

<!-- Success message -->
<div style="background: var(--color-success-50); color: var(--color-success-800);">
  Success! Saved. (Contrast: 7.5:1 ✓ AAA)
</div>
```

## Quality Checklist

Before outputting the final palette, verify:

- [ ] All 50-950 scales generated for primary, secondary, neutral
- [ ] All semantic colors (success, warning, error, info) generated
- [ ] Every color has a calculated "on-{color}" text color
- [ ] All text/background combinations meet specified WCAG level
- [ ] Dark mode variants generated and validated
- [ ] If colorblind-safe enabled: critical pairs verified
- [ ] Requested output format(s) included
- [ ] At least 3 usage examples provided
- [ ] Any WCAG failures clearly flagged with alternatives
- [ ] Accessibility report included for key colors

## Error Handling

### If primary color is too light (lightness > 95%)
```
⚠️ Your primary color (#F8F8F8) is very light and cannot generate
accessible text on white backgrounds. Suggested alternatives:
- Darken to at least 60% lightness
- Use as a background color instead of primary
- Alternative primary: #3B82F6 (meets all standards)
```

### If primary color is too dark (lightness < 10%)
```
⚠️ Your primary color (#0A0A0A) is very dark and cannot generate
accessible light variants. Suggested alternatives:
- Lighten to at least 40% lightness
- Use as neutral-950 instead
- Alternative primary: #3B82F6
```

### If colorblind-safe check fails
```
⚠️ Colorblind Safety Issue:
Your success (#22C55E) and error (#EF4444) colors are not
distinguishable for users with deuteranopia (green-blind).

Suggestions:
- Increase lightness difference (success lighter, error darker)
- Use additional visual cues (icons, patterns) alongside color
- Alternative success: #10B981 (more blue-green)
```

## Advanced Features

### State Variants
Generate hover, active, focus, and disabled states:

```
PRIMARY-500 STATE VARIANTS:
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
Default:  hsl(217, 91%, 50%)
Hover:    hsl(217, 91%, 45%) (10% darker)
Active:   hsl(217, 91%, 40%) (20% darker)
Focus:    hsl(217, 91%, 50%) + 3px ring at 217, 91%, 70%
Disabled: hsl(217, 30%, 70%) (reduced saturation + lightened)
```

### Gradient Combinations
Provide accessible gradient suggestions:

```
ACCESSIBLE GRADIENTS:
━━━━━━━━━━━━━━━━━━━━━━━━
Linear (left to right):
  primary-500 → primary-700
  Contrast: Maintains 4.5:1 with white text throughout

Radial (center):
  primary-400 (center) → primary-600 (edges)
  Contrast: White text safe across entire gradient
```

### Opacity Variants
For overlays and glass morphism:

```
PRIMARY-500 OPACITY VARIANTS:
━━━━━━━━━━━━━━━━━━━━━━━━━━━
95%: hsla(217, 91%, 50%, 0.95) - Nearly opaque
90%: hsla(217, 91%, 50%, 0.90) - Subtle transparency
80%: hsla(217, 91%, 50%, 0.80) - Modal overlays
50%: hsla(217, 91%, 50%, 0.50) - Glass effects
20%: hsla(217, 91%, 50%, 0.20) - Hover states
10%: hsla(217, 91%, 50%, 0.10) - Subtle backgrounds

⚠️ Note: Opacity affects contrast. Re-validate text colors
when using transparent backgrounds over images.
```

## Response Template

Structure your response as follows:

1. **Palette Overview**
   - Base color received
   - Style applied
   - WCAG level targeted
   - Colorblind-safe status

2. **Color Scales**
   - Primary (full 50-950 scale)
   - Secondary (full scale)
   - Success (key values: 50, 500, 700, 900)
   - Warning (key values)
   - Error (key values)
   - Info (key values)
   - Neutral (full scale)

3. **Accessibility Report**
   - Contrast ratios for critical combinations
   - WCAG compliance status
   - Flagged issues and alternatives

4. **Code Export**
   - Format(s) requested
   - Copy-paste ready

5. **Usage Examples**
   - 3-5 practical combinations
   - With contrast ratios noted

6. **Recommendations**
   - Best practices for this palette
   - Warnings about specific combinations
   - Suggestions for edge cases

## Tone and Communication

- Be informative but concise
- Clearly flag accessibility issues with ⚠️ or ✗
- Celebrate successful combinations with ✓
- Provide actionable alternatives, not just problems
- Use visual separators (━━━) for readability
- Include both technical values (HSL) and hex for versatility

This skill should empower designers and developers to create beautiful, accessible interfaces with confidence.
