# UI Component Generator - Skill Instructions

## Purpose
This skill enables AI assistants to generate production-ready UI components from natural language descriptions with clean, accessible code in multiple output formats (Vanilla HTML/CSS, Tailwind CSS, or React TypeScript).

## Skill Capabilities

### Supported Component Types
1. **Buttons**: primary, secondary, ghost, icon, loading states
2. **Cards**: content, pricing, profile, feature, testimonial
3. **Forms**: input, textarea, select, checkbox, radio, switch
4. **Modals/Dialogs**: alert, confirm, form, fullscreen
5. **Navigation**: navbar, sidebar, breadcrumbs, tabs, pagination
6. **Alerts/Notifications**: info, success, warning, error, toast
7. **Empty States**: no data, error, search, onboarding
8. **Loading States**: spinner, skeleton, progress bar, shimmer

### Output Formats
- **Vanilla HTML/CSS**: Pure HTML with CSS custom properties for theming
- **Tailwind CSS**: HTML with Tailwind utility classes
- **React TypeScript**: Complete React components with TypeScript types

## Core Requirements for Every Component

### 1. Accessibility (WCAG 2.1 AA)
- Use semantic HTML5 elements (button, nav, main, article, aside, dialog)
- Include appropriate ARIA attributes:
  - `aria-label` for icon-only buttons
  - `aria-describedby` for form field errors
  - `aria-expanded` for disclosure widgets
  - `aria-modal="true"` for modal dialogs
  - `role="alert"` for error messages
  - `role="status"` for live updates
- Implement keyboard navigation:
  - Tab/Shift+Tab for focus management
  - Enter/Space for button activation
  - Escape to close modals
  - Arrow keys for navigation menus/tabs
- Provide visible focus indicators (outline or focus ring)
- Include screen reader text where needed (sr-only class)

### 2. Visual States
- **Default**: Base appearance
- **Hover**: Cursor pointer, background/border color change
- **Active/Pressed**: Visual feedback on click
- **Focus**: Visible outline or ring (keyboard users)
- **Disabled**: Reduced opacity, cursor not-allowed, no interaction
- **Loading**: Spinner or skeleton display (where applicable)

### 3. Responsive Design
- Mobile-first approach (min-width media queries)
- Touch-friendly tap targets (44px × 44px minimum)
- Responsive typography using clamp() or fluid scale
- Standard breakpoints:
  - sm: 640px
  - md: 768px
  - lg: 1024px
  - xl: 1280px

### 4. Dark Mode Support
Choose approach based on output format:
- **Vanilla CSS**: `@media (prefers-color-scheme: dark)` or data-theme attribute
- **Tailwind CSS**: `dark:` variant classes
- **React**: Theme context or CSS-in-JS theme provider

Ensure proper contrast ratios in both light and dark modes (4.5:1 for body text, 3:1 for large text).

## Generation Workflow

### Step 1: Clarify Requirements
When user requests a component, ask:
1. "Which output format do you prefer: Vanilla HTML/CSS, Tailwind CSS, or React TypeScript?"
2. "Do you need dark mode support?" (default: yes)
3. "Any specific color scheme or brand colors?" (default: blue/neutral palette)
4. "Are there special variants or behaviors needed?"

### Step 2: Generate Component Code

#### Vanilla HTML/CSS Structure
```html
<!-- Semantic HTML with ARIA attributes -->
<button class="btn btn-primary" aria-label="Descriptive label">
  Button Text
</button>

<style>
/* CSS Custom Properties for Theming */
:root {
  /* Colors */
  --btn-primary-bg: #3b82f6;
  --btn-primary-hover: #2563eb;
  --btn-primary-active: #1d4ed8;
  --btn-text: #ffffff;

  /* Spacing */
  --space-2: 0.5rem;
  --space-4: 1rem;

  /* Radius */
  --radius-md: 0.5rem;

  /* Shadows */
  --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1);
}

/* Dark Mode */
@media (prefers-color-scheme: dark) {
  :root {
    --btn-primary-bg: #60a5fa;
    --btn-primary-hover: #3b82f6;
    --btn-primary-active: #2563eb;
  }
}

/* Base Button Styles */
.btn {
  display: inline-flex;
  align-items: center;
  justify-content: center;
  padding: var(--space-2) var(--space-4);
  font-weight: 500;
  border-radius: var(--radius-md);
  border: none;
  cursor: pointer;
  transition: all 0.2s ease;
  min-height: 44px; /* Touch-friendly */
}

.btn:focus-visible {
  outline: 2px solid var(--btn-primary-bg);
  outline-offset: 2px;
}

.btn:disabled {
  opacity: 0.5;
  cursor: not-allowed;
}

/* Primary Variant */
.btn-primary {
  background: var(--btn-primary-bg);
  color: var(--btn-text);
  box-shadow: var(--shadow-md);
}

.btn-primary:hover:not(:disabled) {
  background: var(--btn-primary-hover);
}

.btn-primary:active:not(:disabled) {
  background: var(--btn-primary-active);
}
</style>
```

#### Tailwind CSS Structure
```html
<button
  class="
    inline-flex items-center justify-center
    px-4 py-2 min-h-[44px]
    bg-blue-600 hover:bg-blue-700 active:bg-blue-800
    text-white font-medium
    rounded-lg shadow-md
    focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
    disabled:opacity-50 disabled:cursor-not-allowed
    dark:bg-blue-500 dark:hover:bg-blue-600 dark:active:bg-blue-700
    transition-colors duration-200
  "
  aria-label="Descriptive label"
>
  Button Text
</button>
```

#### React TypeScript Structure
```typescript
import React from 'react';

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'primary' | 'secondary' | 'ghost';
  size?: 'sm' | 'md' | 'lg';
  isLoading?: boolean;
  leftIcon?: React.ReactNode;
  rightIcon?: React.ReactNode;
}

export const Button: React.FC<ButtonProps> = ({
  variant = 'primary',
  size = 'md',
  isLoading = false,
  leftIcon,
  rightIcon,
  children,
  disabled,
  className = '',
  ...props
}) => {
  const baseStyles = `
    inline-flex items-center justify-center
    font-medium rounded-lg
    transition-colors duration-200
    focus:outline-none focus:ring-2 focus:ring-offset-2
  `;

  const variantStyles = {
    primary: `
      bg-blue-600 hover:bg-blue-700 active:bg-blue-800
      text-white
      focus:ring-blue-500
      dark:bg-blue-500 dark:hover:bg-blue-600
    `,
    secondary: `
      bg-gray-200 hover:bg-gray-300 active:bg-gray-400
      text-gray-900
      focus:ring-gray-500
      dark:bg-gray-700 dark:hover:bg-gray-600 dark:text-white
    `,
    ghost: `
      bg-transparent hover:bg-gray-100 active:bg-gray-200
      text-gray-700
      focus:ring-gray-500
      dark:hover:bg-gray-800 dark:text-gray-300
    `
  };

  const sizeStyles = {
    sm: 'px-3 py-1.5 text-sm min-h-[36px]',
    md: 'px-4 py-2 text-base min-h-[44px]',
    lg: 'px-6 py-3 text-lg min-h-[52px]'
  };

  const disabledStyles = (disabled || isLoading) ? 'opacity-50 cursor-not-allowed' : '';

  return (
    <button
      className={`${baseStyles} ${variantStyles[variant]} ${sizeStyles[size]} ${disabledStyles} ${className}`.trim().replace(/\s+/g, ' ')}
      disabled={disabled || isLoading}
      {...props}
    >
      {isLoading ? (
        <span className="flex items-center gap-2">
          <svg
            className="animate-spin h-4 w-4"
            xmlns="http://www.w3.org/2000/svg"
            fill="none"
            viewBox="0 0 24 24"
            aria-hidden="true"
          >
            <circle
              className="opacity-25"
              cx="12"
              cy="12"
              r="10"
              stroke="currentColor"
              strokeWidth="4"
            />
            <path
              className="opacity-75"
              fill="currentColor"
              d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
            />
          </svg>
          <span>Loading...</span>
        </span>
      ) : (
        <>
          {leftIcon && <span className="mr-2" aria-hidden="true">{leftIcon}</span>}
          {children}
          {rightIcon && <span className="ml-2" aria-hidden="true">{rightIcon}</span>}
        </>
      )}
    </button>
  );
};

Button.displayName = 'Button';
```

### Step 3: Provide Usage Examples
Include 2-3 practical examples:

```typescript
// Example 1: Basic Usage
<Button>Click Me</Button>

// Example 2: With Variants
<Button variant="secondary" size="lg">
  Large Secondary Button
</Button>

// Example 3: Loading State
<Button isLoading disabled>
  Submitting...
</Button>

// Example 4: With Icons
<Button leftIcon={<CheckIcon />}>
  Confirm
</Button>
```

### Step 4: Customization Guide
Explain what users can customize:

**For Vanilla CSS:**
- Modify CSS custom properties in `:root`
- Adjust spacing, colors, shadows, border radius
- Add new variants by extending `.btn` base class

**For Tailwind:**
- Update `tailwind.config.js` for theme colors
- Use arbitrary values for one-off customizations
- Extend with custom utility classes

**For React:**
- Add new props to interface
- Extend `variantStyles` or `sizeStyles` objects
- Pass custom className for overrides
- Compose with other components

### Step 5: Accessibility Notes
Document keyboard shortcuts and screen reader behavior:

**Keyboard Navigation:**
- `Tab`: Focus the button
- `Enter` or `Space`: Activate the button
- `Shift+Tab`: Move focus to previous element

**Screen Reader Behavior:**
- Announces as "Button, [button text]"
- If `aria-label` present, announces that instead
- Loading state announces "Loading..." when isLoading is true
- Disabled buttons are skipped or announced as "dimmed" or "unavailable"

## Design Token Conventions

Use consistent naming patterns for CSS custom properties:

```css
/* Color Tokens */
--color-primary-50 through --color-primary-950  /* 50-950 scale */
--color-bg-base, --color-bg-subtle, --color-bg-muted
--color-text-primary, --color-text-secondary, --color-text-tertiary
--color-border-default, --color-border-hover, --color-border-focus

/* Spacing Tokens (4px base) */
--space-1: 0.25rem;  /* 4px */
--space-2: 0.5rem;   /* 8px */
--space-4: 1rem;     /* 16px */
--space-6: 1.5rem;   /* 24px */
/* ... up to --space-20 */

/* Typography Tokens */
--text-xs: 0.75rem;
--text-sm: 0.875rem;
--text-base: 1rem;
--text-lg: 1.125rem;
--text-xl: 1.25rem;
/* ... up to --text-4xl */

/* Shadow Tokens */
--shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
--shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1);
--shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1);
--shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1);

/* Radius Tokens */
--radius-none: 0;
--radius-sm: 0.25rem;
--radius-md: 0.5rem;
--radius-lg: 0.75rem;
--radius-xl: 1rem;
--radius-full: 9999px;

/* Duration Tokens */
--duration-fast: 150ms;
--duration-base: 200ms;
--duration-slow: 300ms;
```

## Component Pattern Library

### Button with Icon
```html
<button class="btn btn-primary">
  <svg class="icon" aria-hidden="true"><!-- icon path --></svg>
  <span>Button Text</span>
</button>
```

### Card Component
```html
<article class="card">
  <img src="image.jpg" alt="Descriptive alt text" loading="lazy" class="card-image">
  <div class="card-content">
    <h3 class="card-title">Card Title</h3>
    <p class="card-description">Card description text</p>
    <button class="card-action">Action</button>
  </div>
</article>
```

### Modal Dialog
```html
<dialog class="modal" role="dialog" aria-labelledby="modal-title" aria-modal="true">
  <div class="modal-content">
    <div class="modal-header">
      <h2 id="modal-title">Modal Title</h2>
      <button class="modal-close" aria-label="Close modal">
        <svg aria-hidden="true"><!-- close icon --></svg>
      </button>
    </div>
    <div class="modal-body">
      <!-- Modal content -->
    </div>
    <div class="modal-footer">
      <button class="btn btn-secondary">Cancel</button>
      <button class="btn btn-primary">Confirm</button>
    </div>
  </div>
  <div class="modal-backdrop" aria-hidden="true"></div>
</dialog>
```

### Form Input with Validation
```html
<div class="form-field">
  <label for="email" class="form-label">Email Address</label>
  <input
    id="email"
    type="email"
    class="form-input"
    aria-describedby="email-error"
    aria-invalid="true"
    placeholder="you@example.com"
  >
  <span id="email-error" class="form-error" role="alert">
    Please enter a valid email address
  </span>
</div>
```

### Navigation Bar
```html
<nav class="navbar" role="navigation" aria-label="Main navigation">
  <div class="navbar-brand">
    <a href="/" aria-label="Home">
      <img src="logo.svg" alt="Company Logo">
    </a>
  </div>
  <ul class="navbar-menu">
    <li><a href="/products" class="navbar-link">Products</a></li>
    <li><a href="/about" class="navbar-link">About</a></li>
    <li><a href="/contact" class="navbar-link">Contact</a></li>
  </ul>
</nav>
```

### Alert/Notification
```html
<div class="alert alert-success" role="alert">
  <svg class="alert-icon" aria-hidden="true"><!-- success icon --></svg>
  <div class="alert-content">
    <h3 class="alert-title">Success!</h3>
    <p class="alert-message">Your changes have been saved.</p>
  </div>
  <button class="alert-close" aria-label="Dismiss alert">
    <svg aria-hidden="true"><!-- close icon --></svg>
  </button>
</div>
```

### Empty State
```html
<div class="empty-state">
  <svg class="empty-state-icon" aria-hidden="true"><!-- illustration --></svg>
  <h2 class="empty-state-title">No results found</h2>
  <p class="empty-state-description">Try adjusting your search filters</p>
  <button class="btn btn-primary">Clear Filters</button>
</div>
```

### Skeleton Loader
```html
<div class="skeleton-card" aria-busy="true" aria-label="Loading content">
  <div class="skeleton skeleton-image"></div>
  <div class="skeleton skeleton-text"></div>
  <div class="skeleton skeleton-text" style="width: 60%;"></div>
</div>

<style>
.skeleton {
  background: linear-gradient(
    90deg,
    #f0f0f0 25%,
    #e0e0e0 50%,
    #f0f0f0 75%
  );
  background-size: 200% 100%;
  animation: shimmer 1.5s infinite;
}

@keyframes shimmer {
  0% { background-position: 200% 0; }
  100% { background-position: -200% 0; }
}

@media (prefers-reduced-motion: reduce) {
  .skeleton {
    animation: none;
    background: #f0f0f0;
  }
}
</style>
```

## Quality Assurance Checklist

Before delivering component code, verify:

### Accessibility
- [ ] Semantic HTML elements used appropriately
- [ ] ARIA attributes present where needed (not overused)
- [ ] Keyboard navigation fully functional
- [ ] Focus indicators visible on all interactive elements
- [ ] Color contrast meets WCAG AA (4.5:1 body, 3:1 large text)
- [ ] Screen reader announcements are meaningful

### Visual Design
- [ ] Hover states defined for interactive elements
- [ ] Active/pressed states provide visual feedback
- [ ] Focus states distinct from hover states
- [ ] Disabled states clearly indicate non-interactivity
- [ ] Loading states animate smoothly (respect prefers-reduced-motion)

### Responsive Behavior
- [ ] Touch targets meet 44px × 44px minimum
- [ ] Layout adapts gracefully at all breakpoints
- [ ] Typography scales appropriately
- [ ] Spacing remains proportional on mobile
- [ ] No horizontal scroll at any viewport width

### Dark Mode
- [ ] Dark mode styles included (CSS media query or Tailwind dark:)
- [ ] Colors maintain proper contrast in dark mode
- [ ] Shadows adjusted for dark backgrounds
- [ ] Border colors visible on dark backgrounds

### Code Quality
- [ ] No console errors or warnings
- [ ] Code is copy-paste ready (no placeholders or TODOs)
- [ ] Consistent naming conventions followed
- [ ] Comments explain complex logic
- [ ] TypeScript types are accurate (for React components)

## Response Format Template

Structure all component responses using this format:

```markdown
# [Component Name]

## Preview
[Describe what the component looks like and its key visual features]

## Code
[Provide complete, copy-paste ready code]

## Usage Examples
[Show 2-3 practical usage scenarios]

## Customization Options
[Explain what can be customized and how:
- Colors (which CSS variables or Tailwind classes)
- Sizing (padding, font sizes, etc.)
- Behavior (props, variants)
]

## Accessibility Notes
**Keyboard Navigation:**
- [List keyboard shortcuts]

**Screen Reader Behavior:**
- [Describe what screen readers announce]

**Focus Management:**
- [Explain focus behavior]
```

## Anti-Patterns to Avoid

### HTML Anti-Patterns
- ❌ Using `<div onclick>` instead of `<button>`
- ❌ Using `<div>` for everything (div soup)
- ❌ Missing `alt` attributes on images
- ❌ Using `placeholder` as a label replacement
- ❌ Nested interactive elements (button inside link)

### Accessibility Anti-Patterns
- ❌ Missing ARIA labels on icon-only buttons
- ❌ Using `aria-label` on non-interactive elements
- ❌ Redundant ARIA (aria-label on button with visible text)
- ❌ Low color contrast ratios (<4.5:1)
- ❌ No visible focus indicators
- ❌ Click handlers without keyboard handlers
- ❌ Missing form labels or using placeholder-only labels

### CSS Anti-Patterns
- ❌ Hardcoded colors without dark mode consideration
- ❌ Using `!important` excessively
- ❌ Inline styles instead of classes
- ❌ Non-responsive units (px for everything)
- ❌ Missing transition for state changes
- ❌ Touch targets smaller than 44px

### React Anti-Patterns
- ❌ Missing TypeScript types or using `any`
- ❌ Not extending native HTML element props
- ❌ Mutating props directly
- ❌ Missing displayName on components
- ❌ Using index as key in lists
- ❌ Not memoizing expensive computations

## Advanced Patterns

### Compound Components (React)
```typescript
interface TabsContextValue {
  activeTab: string;
  setActiveTab: (tab: string) => void;
}

const TabsContext = React.createContext<TabsContextValue | undefined>(undefined);

export const Tabs: React.FC<{ defaultTab: string; children: React.ReactNode }> & {
  List: typeof TabsList;
  Tab: typeof TabsTab;
  Panel: typeof TabsPanel;
} = ({ defaultTab, children }) => {
  const [activeTab, setActiveTab] = React.useState(defaultTab);

  return (
    <TabsContext.Provider value={{ activeTab, setActiveTab }}>
      <div className="tabs">{children}</div>
    </TabsContext.Provider>
  );
};

Tabs.List = TabsList;
Tabs.Tab = TabsTab;
Tabs.Panel = TabsPanel;

// Usage:
<Tabs defaultTab="tab1">
  <Tabs.List>
    <Tabs.Tab id="tab1">Tab 1</Tabs.Tab>
    <Tabs.Tab id="tab2">Tab 2</Tabs.Tab>
  </Tabs.List>
  <Tabs.Panel id="tab1">Content 1</Tabs.Panel>
  <Tabs.Panel id="tab2">Content 2</Tabs.Panel>
</Tabs>
```

### Polymorphic Components (React)
```typescript
type PolymorphicProps<E extends React.ElementType> = {
  as?: E;
  children: React.ReactNode;
} & Omit<React.ComponentPropsWithoutRef<E>, 'as'>;

export const Box = <E extends React.ElementType = 'div'>({
  as,
  children,
  ...props
}: PolymorphicProps<E>) => {
  const Component = as || 'div';
  return <Component {...props}>{children}</Component>;
};

// Usage:
<Box as="button" onClick={() => {}}>Click</Box>
<Box as="a" href="/about">About</Box>
```

## Example Interaction Flow

**User Request:**
> "Create a toast notification component in React TypeScript"

**AI Response:**
```
I'll create a toast notification component with multiple variants (info, success, warning, error).

Quick questions:
1. Do you need auto-dismiss functionality? (default: yes, 5 seconds)
2. Should toasts stack or replace each other? (default: stack)
3. Position preference? (default: bottom-right)

Proceeding with defaults...

# Toast Notification Component

## Preview
A toast notification that slides in from the bottom-right corner with smooth animations. Includes icon, title, message, and close button. Auto-dismisses after 5 seconds but can be manually closed. Supports info (blue), success (green), warning (yellow), and error (red) variants.

## Code
[Complete React TypeScript component with all states, animations, and accessibility]

## Usage Examples
[3 examples showing different variants and use cases]

## Customization Options
[How to customize duration, position, animation, colors]

## Accessibility Notes
[Keyboard navigation, ARIA live region, screen reader behavior]
```

## Testing Recommendations

Suggest these testing approaches when delivering components:

### Manual Testing
- Test with keyboard only (no mouse)
- Test with screen reader (VoiceOver, NVDA, JAWS)
- Test at various viewport sizes (mobile, tablet, desktop)
- Test in both light and dark mode
- Test with browser zoom at 200%

### Automated Testing
```typescript
// Example Jest + Testing Library test for Button
import { render, screen, fireEvent } from '@testing-library/react';
import { Button } from './Button';

describe('Button', () => {
  it('renders children correctly', () => {
    render(<Button>Click Me</Button>);
    expect(screen.getByText('Click Me')).toBeInTheDocument();
  });

  it('handles click events', () => {
    const handleClick = jest.fn();
    render(<Button onClick={handleClick}>Click</Button>);
    fireEvent.click(screen.getByText('Click'));
    expect(handleClick).toHaveBeenCalledTimes(1);
  });

  it('shows loading state', () => {
    render(<Button isLoading>Submit</Button>);
    expect(screen.getByText('Loading...')).toBeInTheDocument();
  });

  it('is accessible', async () => {
    const { container } = render(<Button>Accessible Button</Button>);
    const results = await axe(container);
    expect(results).toHaveNoViolations();
  });
});
```

## Performance Considerations

### Vanilla CSS
- Use CSS containment for isolated components
- Minimize selector specificity
- Avoid expensive properties (box-shadow, filter) on animating elements
- Use `will-change` sparingly for critical animations

### Tailwind CSS
- Enable JIT mode for faster builds
- Purge unused styles in production
- Use `@apply` sparingly (increases bundle size)

### React TypeScript
- Memoize expensive computations with `useMemo`
- Prevent unnecessary re-renders with `React.memo`
- Use `useCallback` for event handlers in memoized components
- Lazy load heavy components with `React.lazy`

## Browser Support
Target modern browsers (last 2 versions):
- Chrome/Edge (Chromium)
- Firefox
- Safari
- Mobile Safari
- Chrome Android

Use progressive enhancement for older browsers.

## Resources and References

When generating components, mentally reference these standards:
- WCAG 2.1 Level AA guidelines
- ARIA Authoring Practices Guide (APG)
- MDN Web Docs for HTML semantics
- React TypeScript Cheatsheet
- Tailwind CSS documentation

## Conclusion

This skill empowers AI assistants to generate production-ready, accessible, responsive UI components that follow modern web development best practices. Every component should be immediately usable with no modifications required, while remaining easily customizable for specific project needs.

Always prioritize accessibility, semantic HTML, and user experience over visual aesthetics alone. The best components are those that work for everyone, regardless of how they access the web.
