---
name: api-security-tester
version: 1.0.0
description: API security testing skill based on OWASP API Security Top 10 (2023) for REST, GraphQL, and gRPC APIs with test case generation and remediation guidance
author: FindSkill.ai
license: MIT
---

# Initialization

## Overview

This skill provides a comprehensive API security testing framework based on the OWASP API Security Top 10 (2023 edition). It covers REST, GraphQL, and gRPC APIs with specific test cases, request/response examples, and code-level remediation guidance.

**What this skill enables:**
- Test APIs against all 10 OWASP API Security Top 10 (2023) categories
- Generate structured test cases with HTTP request/response examples
- Review API designs for security anti-patterns before implementation
- Test authentication mechanisms (JWT, OAuth2, API keys, mTLS)
- Identify authorization flaws (BOLA, BFLA, mass assignment)
- Assess GraphQL-specific vulnerabilities (introspection, depth, batching)
- Assess gRPC-specific vulnerabilities (metadata injection, reflection)
- Integrate security testing into CI/CD pipelines
- Generate professional security test reports

**Time to initialize:** ~1 minute (prompt-only skill, no scripts required)

## Directory Structure

```
api-security-tester/
├── SKILL.md          # Main API security testing instructions and test cases
└── INIT.md           # This initialization manifest
```

## Dependencies

This is a prompt-only skill with no hard dependencies. However, the skill references these tools for hands-on testing:

**API Security Scanning:**
```bash
# OWASP ZAP (automated API scanning)
docker pull owasp/zap2docker-stable

# Nuclei (template-based scanning)
go install -v github.com/projectdiscovery/nuclei/v3/cmd/nuclei@latest

# kiterunner (API endpoint discovery)
go install github.com/assetnote/kiterunner/cmd/kr@latest
```

**JWT Testing:**
```bash
# jwt_tool (JWT vulnerability testing)
pip install jwt-tool

# hashcat (JWT secret brute force)
# Available at: https://hashcat.net/hashcat/
```

**gRPC Testing:**
```bash
# grpcurl (gRPC command-line client)
brew install grpcurl
# or: go install github.com/fullstorydev/grpcurl/cmd/grpcurl@latest
```

**General API Testing:**
```bash
# Burp Suite Community (manual testing)
# Download from: https://portswigger.net/burp/communitydownload

# Postman (API testing collections)
# Download from: https://www.postman.com/downloads/
```

## Files to Generate

This is a **prompt-only skill** - no scripts or additional files need to be generated. The complete API security testing framework is contained in SKILL.md.

## Post-Init Steps

### For Claude Code Users

```bash
# Copy to your Claude Code skills directory
mkdir -p ~/.claude/skills/api-security-tester
cp SKILL.md INIT.md ~/.claude/skills/api-security-tester/
```

Then reference the skill when discussing API security testing.

### For Other AI Assistants

1. Open `SKILL.md`
2. Copy the entire content after the frontmatter (after the second `---`)
3. Paste into your AI assistant's system prompt or conversation context
4. Start your API security testing conversation

## Quick Start Examples

### Example 1: Full OWASP API Security Audit

```
User: I have a REST API for an e-commerce platform. It uses JWT
      authentication and has endpoints for users, orders, payments,
      and admin operations. Can you run a full OWASP API Security
      Top 10 assessment?

AI: I'll conduct a comprehensive API security audit against all
    OWASP API Security Top 10 (2023) categories. Let me start
    with the most critical areas.

    Starting with API1:2023 - Broken Object Level Authorization:

    TEST-BOLA-001: GET /api/v1/orders/{order_id}
    Can User A access User B's order by changing the order_id?

    TEST-BOLA-002: PUT /api/v1/users/{user_id}
    Can User A modify User B's profile?

    [Continues through all 10 categories with specific test cases
    for the e-commerce context...]
```

### Example 2: GraphQL Security Assessment

```
User: Audit my GraphQL API for security issues. It has queries
      for users, products, and orders with mutations for CRUD
      operations. Authentication is via JWT in the Authorization header.

AI: For GraphQL APIs, I'll check standard OWASP categories plus
    GraphQL-specific attack vectors:

    TEST-GQL-001: Introspection enabled?
    POST /graphql
    {"query": "{ __schema { types { name fields { name } } } }"}

    TEST-GQL-002: Query depth attack
    {"query": "{ users { orders { items { product { reviews { ... } } } } } }"}

    TEST-GQL-004: Batch brute force
    [{"query": "mutation { login(password:\"pass1\") { token } }"},
     {"query": "mutation { login(password:\"pass2\") { token } }"},
     ...]

    [Continues with all GraphQL-specific and general API tests...]
```

### Example 3: API Design Review (Shift Left)

```
User: I'm designing a new API. Here's my OpenAPI spec. Can you
      review it for security issues before I start building?

AI: Great approach - catching security issues at design time is
    10x cheaper than fixing them in production. Let me review
    your spec against OWASP API Security Top 10:

    FINDING-001: No authentication on /api/v1/users endpoint
    Recommendation: Add OAuth2 security scheme

    FINDING-002: User object schema includes "role" as writable
    Risk: Mass assignment (API3:2023)
    Recommendation: Separate read/write schemas

    FINDING-003: No pagination parameters on list endpoints
    Risk: Resource exhaustion (API4:2023)
    Recommendation: Add limit/offset with max page size

    [Continues with full design review...]
```

### Example 4: CI/CD Integration

```
User: How do I integrate API security testing into my GitHub Actions
      CI/CD pipeline?

AI: Here's a GitHub Actions workflow that runs API security tests
    on every PR that touches API code:

    [Provides complete workflow YAML with OWASP ZAP, Nuclei,
    and custom BOLA test integration...]
```

## Compatibility

**Tested with:**
- Claude (claude.ai, Claude Code)
- ChatGPT (GPT-4, GPT-4 Turbo, GPT-4o)
- Gemini (Gemini Pro, Gemini Ultra)
- GitHub Copilot Chat
- Llama 3 (70B+)
- Mistral Large

**Works best with:** Claude and GPT-4 for comprehensive security analysis with code-level remediation.

## Variables

Customize these placeholders for your specific API:

| Variable | Default | Description |
|----------|---------|-------------|
| `{{api_spec}}` | (paste spec here) | Your OpenAPI spec, GraphQL schema, or endpoint description |
| `{{api_type}}` | `REST` | REST, GraphQL, or gRPC |
| `{{auth_method}}` | `JWT Bearer Token` | JWT, OAuth2, API Keys, mTLS, SAML |
| `{{testing_scope}}` | `full` | "full" or specific OWASP category (e.g., "API1 - BOLA") |
| `{{endpoint_description}}` | (describe endpoints) | HTTP methods, paths, expected behavior |

## Key Sections in SKILL.md

1. **OWASP API Security Top 10 (2023)** - All 10 categories with specific test cases, request/response examples, and remediation code

2. **GraphQL-Specific Tests** - Introspection, query depth, complexity, batching, field suggestions, injection

3. **gRPC-Specific Tests** - Metadata injection, server reflection, message size, stream exhaustion

4. **Authentication Security** - JWT checklist, OAuth2 checklist, API key checklist, mTLS checklist

5. **Authorization Patterns** - RBAC testing matrix, ABAC testing, field-level authorization

6. **Input Validation** - SQL/NoSQL/command injection, mass assignment, type confusion

7. **Tools Integration** - Burp Suite, OWASP ZAP, Postman, Nuclei, jwt_tool, grpcurl

8. **CI/CD Integration** - GitHub Actions workflow, pre-commit hooks

9. **Test Report Template** - Professional report format with findings and remediation priority

## Recommended Workflow

1. **Gather API information** - Specification, auth method, endpoints, business context
2. **Start with BOLA (API1)** - Most common and critical API vulnerability
3. **Test authentication (API2)** - JWT/OAuth2 weaknesses
4. **Check authorization (API3, API5)** - Mass assignment and function-level access
5. **Assess resource limits (API4)** - Rate limiting, pagination, request sizes
6. **Test for SSRF (API7)** - If API fetches external URLs
7. **Review configuration (API8)** - CORS, headers, TLS, error messages
8. **Protocol-specific tests** - GraphQL or gRPC specific vulnerabilities
9. **Generate report** - Prioritized findings with remediation guidance
10. **Integrate into CI/CD** - Automate recurring security checks

## Additional Resources

- OWASP API Security Top 10 (2023): https://owasp.org/API-Security/editions/2023/en/0x11-t10/
- OWASP Testing Guide: https://owasp.org/www-project-web-security-testing-guide/
- PortSwigger Web Academy: https://portswigger.net/web-security/api-testing
- OWASP REST Security Cheat Sheet: https://cheatsheetseries.owasp.org/cheatsheets/REST_Security_Cheat_Sheet.html
- OWASP GraphQL Cheat Sheet: https://cheatsheetseries.owasp.org/cheatsheets/GraphQL_Cheat_Sheet.html

---

Downloaded from [Find Skill.ai](https://findskill.ai)
